% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extractChromMetricsFunctions.R
\name{extractChromMetrics}
\alias{extractChromMetrics}
\title{Extract metrics of chromatographic peak quality}
\usage{
extractChromMetrics(
  peak_data,
  recalc_betas = FALSE,
  ms1_data = NULL,
  verbosity = 0
)
}
\arguments{
\item{peak_data}{Flat-form XC-MS data with columns for the bounding box of
a chromatographic peak (mzmin, mzmax, rtmin, rtmax) as grouped by a feature
ID. Must be provided WITHOUT retention time correction for proper matching
to the values in the raw data.}

\item{recalc_betas}{Scalar boolean controlling whether existing beta values
(as calculated in the XCMS object when peakpicked with \code{CentWaveParam(verboseBetaColumns=TRUE))}
should be used as-is (the default) or recalculated. See
\url{https://github.com/sneumann/xcms/pull/685} for differences in these
implementations.}

\item{ms1_data}{Optional data.table object produced by RaMS containing MS1
data with columns for filename, rt, mz, and int. If not provided, the files
are detected from the filepath column in peak_data.}

\item{verbosity}{Scalar value between zero and two determining how much
diagnostic information is produced. 0 should return nothing, 1 should
return text-based progress markers, and 2 will return diagnostic plots if
available.}
}
\value{
A data.frame containing one row for each feature in peak_data, with
columns containing the median peak shape value (med_cor), and the median SNR
value (med_snr).
}
\description{
This function takes flat-form XC-MS data (i.e. the format produced by
`makeXcmsObjectFlat`) and calculates metrics of peak shape and
similarity for downstream processing (e.g. by \code{\link{updateXcmsObjFeats}}).
The core metrics are those described in \url{https://doi.org/10.1186/s12859-023-05533-4},
corresponding to peak shape (correlation coefficient between the raw data
and an idealized bell curve, beta_cor) and within-peak signal-to-noise (
maximum intensity divided by the standard deviation of the residuals after
the best-fitting bell is subtracted from the raw data). Additionally,
the function interpolates the raw data to fixed retention time intervals
and performs a principal components analysis (PCA) across a file-by-RT matrix
which typically extracts good-looking peaks in the first component (PC1).
These functions are calculated on the raw MS data as obtained via
\pkg{RaMS}, so either the filepaths must be included in the peak_data object
or supplied as ms1_data.
}
\examples{
library(xcms)
library(MSnbase)
library(dplyr)
mzML_files <- system.file("extdata", package = "RaMS") \%>\%
    list.files(full.names = TRUE, pattern = "[A-F].mzML")
register(BPPARAM = SerialParam())
cwp <- CentWaveParam(snthresh = 0, extendLengthMSW = TRUE, integrate = 2)
obp <- ObiwarpParam(binSize = 0.1, response = 1, distFun = "cor_opt")
pdp <- PeakDensityParam(
    sampleGroups = 1:3, bw = 12, minFraction = 0,
    binSize = 0.001, minSamples = 0
)
xcms_filled <- mzML_files \%>\%
    readMSData(msLevel. = 1, mode = "onDisk") \%>\%
    findChromPeaks(cwp) \%>\%
    adjustRtime(obp) \%>\%
    groupChromPeaks(pdp) \%>\%
    fillChromPeaks(FillChromPeaksParam(ppm = 5))
peak_data <- makeXcmsObjFlat(xcms_filled)
feat_metrics <- extractChromMetrics(peak_data, recalc_betas = TRUE)
}
