% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/database.R
\name{database}
\alias{database}
\alias{write_hdf5}
\alias{read_hdf5}
\title{Creat databases for the Shiny App}
\usage{
write_hdf5(data, dir = "./data_shm", replace = FALSE)

read_hdf5(file, name = NULL, dir)
}
\arguments{
\item{data}{A nested `list` of each numeric data and aSVG pair. Each pair consists of four slots: `name`, `display`, `data`, and `svg`, e.g. `lis1 <- list(name='mus.brain', display='Mouse brain (SHM)', data='./mus_brain.txt', svg='./mus_brain.svg')`. The `name` contains a syntactically valid entry for R while the `display` contains an entry for display on the user interface. The `data` and `svg` contain paths of numeric data and aSVGs that will be included in the data base respectively. The supported data containers include `data.frame` and `SummarizedExperiment`. See the `data` argument in \link{filter_data} for data formats. After formatted, the data should be saved as ".rds" files with the function \link{saveRDS}, then assign the corresponding paths the `data` slot. By contrast, the `svg` slot contains one or multiple aSVG/raster image file paths. Each numeric data set is first saved in an independent DHF5-based `SummarizedExperiment` (SE) object, then all these saved SE objects and aSVG/raster images are compressed together into a file "data_shm.tar", i.e. the backend database. In addition, the nested `list` assigned to `data` is also included in the database for querying purpose (\link{read_hdf5}).}

\item{dir}{The directory for saving assay data and aSVGs in query.}

\item{replace}{If `TRUE`, the existing database with the same name in `dir` will be replaced.}

\item{file}{The path of a backend database of Shiny App, which is the file of `data_shm.tar` generated by \link{write_hdf5}.}

\item{name}{One or multiple data set names (see the `data` argument in \link{write_hdf5}) in a vector, such as \code{c('test_leaf', 'test_chicken')}. If `match`, the matching `list` between assay data and aSVGs will be returned.}
}
\value{
`write_hdf5`: a file 'data_shm.tar' saved on disk. \cr
`read_hdf5`: a nested `list` of file paths corresponding to data-aSVG pairs.
}
\description{
The function \link{write_hdf5} is designed to construct the a backend database for the Shiny App (\link{shiny_shm}), while \link{read_hdf5} is designed to query the database.
}
\examples{

## Example of a single aSVG instance.

# The example data included in this package come from an RNA-seq analysis on 
# development of 7 chicken organs under 9 time points (Cardoso-Moreira et al. 2019). 
# The complete raw count data are downloaded using the R package ExpressionAtlas
# (Keays 2019) with the accession number "E-MTAB-6769". 
# Access example count data. 
count.chk <- read.table(system.file('extdata/shinyApp/data/count_chicken.txt', 
package='spatialHeatmap'), header=TRUE, row.names=1, sep='\t')
count.chk[1:3, 1:5]

# A targets file describing spatial features and variables is made based on the 
# experiment design.
target.chk <- read.table(system.file('extdata/shinyApp/data/target_chicken.txt', 
package='spatialHeatmap'), header=TRUE, row.names=1, sep='\t')
# Every column in example data 2 corresponds with a row in the targets file. 
target.chk[1:5, ]
# Store example data in "SummarizedExperiment".
library(SummarizedExperiment)
se.chk <- SummarizedExperiment(assay=count.chk, colData=target.chk)

# Indicate spatial features and experiment variables with "spFeature" and "variable"
# in the "colData" slot respectively.
colnames(colData(se.chk))[8] <- 'spFeature'
colnames(colData(se.chk))[6] <- 'variable'
colData(se.chk)[1:2, ]

# Create a temporary directory "db_shm".
dir.db <- file.path(tempdir(check=TRUE), 'db_shm')
if (!dir.exists(dir.db)) dir.create(dir.db)

# Save the assay data in the temporary directory.
chk.dat.pa <- file.path(dir.db, 'test_chicken.rds')
saveRDS(se.chk, file=chk.dat.pa)

# The chicken aSVG downloaded from the EBI aSVG repository (https://github.com/ebi-gene-
# expression-group/anatomogram/tree/master/src/svg) is included in this package and 
# accessed as below.
svg.chk <- system.file("extdata/shinyApp/data", "gallus_gallus.svg",
package="spatialHeatmap")
# Store file paths of data and aSVG pair in a list. 
dat1 <- list(name='test_chicken', display='Test (chicken SHM)', data=chk.dat.pa, svg=svg.chk)

## Example of multiple aSVG and raster images.

# Two aSVG instances of maize leaves represent two growth stages and each aSVG has a raster 
# image counterpart.
# Random numeric data.
pa.leaf <- system.file("extdata/shinyApp/data", 'dat_overlay.txt',
package="spatialHeatmap")
dat.leaf <- read_fr(pa.leaf); dat.leaf[1:2, ]
# Save the assay data in the temporary directory.
leaf.dat.pa <- file.path(dir.db, 'test_leaf.rds')
saveRDS(dat.leaf, file=leaf.dat.pa)

# Paths of the two aSVG files.
svg.leaf1 <- system.file("extdata/shinyApp/data", 'maize_leaf_shm1.svg',
package="spatialHeatmap")
svg.leaf2 <- system.file("extdata/shinyApp/data", 'maize_leaf_shm2.svg',
package="spatialHeatmap") 
# Paths of the two corresponsing raster images.
rst.leaf1 <- system.file("extdata/shinyApp/data", 'maize_leaf_shm1.png',
package="spatialHeatmap")
rst.leaf2 <- system.file("extdata/shinyApp/data", 'maize_leaf_shm2.png',
package="spatialHeatmap") 

# Store file paths of data, aSVG, raster images in a list. 
dat2 <- list(name='test_leaf', display='Test (maize leaf SHM)', data=leaf.dat.pa, 
svg=c(svg.leaf1, svg.leaf2, rst.leaf1, rst.leaf2))

# Store these two data sets in a nested list. 
dat.lis <- list(dat1, dat2) 

\donttest{
# Save the database in the temporary directory. 
write_hdf5(data=dat.lis, dir=dir.db, replace=TRUE)
# Read data-aSVG pair of "test_leaf" from "data_shm.tar".
dat.q <- read_hdf5(file=file.path(dir.db, 'data_shm.tar'), name='test_leaf')
# Numeric data.
dat.leaf <- readRDS(dat.q[[1]]$data)
# aSVG.
svg1 <- dat.q[[1]]$svg[1]; svg2 <- dat.q[[1]]$svg[2]
rst1 <- dat.q[[1]]$svg[3]; rst2 <- dat.q[[1]]$svg[4]
svg.leaf <- read_svg(svg=c(svg1, svg2), raster=c(rst1, rst2))
}

\donttest{  
# Create customized Shiny Apps with the database.
custom_shiny(db=file.path(dir.db, 'data_shm.tar'), app.dir='~/test_shiny')
# Create customized Shiny Apps with the data sets in a nested list.
custom_shiny(data=dat.lis, app.dir='~/test_shiny')
# Run the app.
shiny::runApp('~/test_shiny/shinyApp') 
}
}
\references{
SummarizedExperiment: SummarizedExperiment container. R package version 1.10.1
\cr Pagès H, Lawrence M, Aboyoun P (2023). _S4Vectors: Foundation of vector-like and list-like containers in Bioconductor_. doi:10.18129/B9.bioc.S4Vectors <https://doi.org/10.18129/B9.bioc.S4Vectors>, R package version 0.38.1, <https://bioconductor.org/packages/S4Vectors>.
\cr R Core Team (2018). R: A language and environment for statistical computing. R Foundation for Statistical Computing, Vienna, Austria. URL https://www.R-project.org/
\cr R Core Team (2018). R: A language and environment for statistical computing. R Foundation for Statistical Computing, Vienna, Austria. URL https://www.R-project.org/
\cr Fischer B, Smith M, Pau G (2023). rhdf5: R Interface to HDF5. doi:10.18129/B9.bioc.rhdf5, R package version 2.46.0, https://bioconductor.org/packages/rhdf5
\cr Mustroph, Angelika, M Eugenia Zanetti, Charles J H Jang, Hans E Holtan, Peter P Repetti, David W Galbraith, Thomas Girke, and Julia Bailey-Serres. 2009. “Profiling Translatomes of Discrete Cell Populations Resolves Altered Cellular Priorities During Hypoxia in Arabidopsis.” Proc Natl Acad Sci U S A 106 (44): 18843–8 
\cr Davis, Sean, and Paul Meltzer. 2007. “GEOquery: A Bridge Between the Gene Expression Omnibus (GEO) and BioConductor.” Bioinformatics 14: 1846–7
\cr Gautier, Laurent, Leslie Cope, Benjamin M. Bolstad, and Rafael A. Irizarry. 2004. “Affy—analysis of Affymetrix GeneChip Data at the Probe Level.” Bioinformatics 20 (3). Oxford, UK: Oxford University Press: 307–15. doi:10.1093/bioinformatics/btg405
\cr Keays, Maria. 2019. ExpressionAtlas: Download Datasets from EMBL-EBI Expression Atlas
\cr Huber, W., V. J. Carey, R. Gentleman, S. An ders, M. Carlson, B. S. Carvalho, H. C. Bravo, et al. 2015. “Orchestrating High-Throughput Genomic Analysis Wit H Bioconductor.” Nature Methods 12 (2): 115–21. http://www.nature.com/nmeth/journal/v12/n2/full/nmeth.3252.html
\cr Love, Michael I., Wolfgang Huber, and Simon Anders. 2014. "Moderated Estimation of Fold Change and Dispersion for RNA-Seq Data with DESeq2." Genome Biology 15 (12): 550. doi:10.1186/s13059-014-0550-8
\cr McCarthy, Davis J., Chen, Yunshun, Smyth, and Gordon K. 2012. "Differential Expression Analysis of Multifactor RNA-Seq Experiments with Respect to Biological Variation." Nucleic Acids Research 40 (10): 4288–97
\cr Cardoso-Moreira, Margarida, Jean Halbert, Delphine Valloton, Britta Velten, Chunyan Chen, Yi Shao, Angélica Liechti, et al. 2019. “Gene Expression Across Mammalian Organ Development.” Nature 571 (7766): 505–9
}
\author{
Jianhai Zhang \email{jzhan067@ucr.edu} \cr Dr. Thomas Girke \email{thomas.girke@ucr.edu}
}
