% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/Raw_Methods.R,
%   R/peak_detection.R
\name{detectPeak}
\alias{detectPeak}
\alias{detectPeak,ptrRaw-method}
\alias{detectPeak,ptrSet-method}
\title{Detection and quantification of peaks for a ptrSet object.}
\usage{
detectPeak(
  x,
  ppm = 130,
  minIntensity = 10,
  minIntensityRate = 0.01,
  mzNominal = NULL,
  resolutionRange = NULL,
  fctFit = NULL,
  smoothPenalty = NULL,
  parallelize = FALSE,
  nbCores = 2,
  saving = TRUE,
  saveDir = getParameters(x)$saveDir,
  ...
)

\S4method{detectPeak}{ptrRaw}(
  x,
  ppm = 130,
  minIntensity = 10,
  minIntensityRate = 0.01,
  mzNominal = NULL,
  resolutionRange = NULL,
  fctFit = NULL,
  smoothPenalty = NULL,
  timeLimit,
  knots = NULL,
  mzPrimaryIon = 21.022,
  ...
)

\S4method{detectPeak}{ptrSet}(
  x,
  ppm = 130,
  minIntensity = 10,
  minIntensityRate = 0.01,
  mzNominal = NULL,
  resolutionRange = NULL,
  fctFit = NULL,
  smoothPenalty = 0,
  parallelize = FALSE,
  nbCores = 2,
  saving = TRUE,
  saveDir = getParameters(x)$saveDir,
  ...
)
}
\arguments{
\item{x}{a \code{\link[ptairMS]{ptrSet}} object}

\item{ppm}{minimum distance in ppm between two peaks}

\item{minIntensity}{minimum intensity for peak detection. The final threshold 
for peak detection will be: max(\code{minIntensity}, threshold noise ). 
The threshold noise corresponds to
 max(max(noise around the nominal mass), \code{minIntensityRate} * 
 max(intensity within the nominal mass)}

\item{minIntensityRate}{Fraction of the maximum 
intensity to be used for noise thresholding}

\item{mzNominal}{nominal masses at which peaks will be detected; if \code{NULL}, 
all nominal masses of the mass axis}

\item{resolutionRange}{vector with the minimum, average, and 
maximum resolution of the PTR instrument. If \code{NULL}, the values are 
estimated by using the calibration peaks.}

\item{fctFit}{function for the peak quantification: should be sech2 
or averagePeak. If \code{NULL}, the best function is selected by using the 
calibration peaks}

\item{smoothPenalty}{second order penalty coefficient of the p-spline used 
for two-dimensional regression. If \code{NULL}, the coefficient is estimated by 
generalized cross validation (GCV) criteria}

\item{parallelize}{Boolean. If \code{TRUE}, loops over files are parallelized}

\item{nbCores}{number of cluster to use for parallel computation.}

\item{saving}{boolean. If TRUE, the object will be saved in saveDir with the
\code{setName} parameter of the \code{createPtrSet} function}

\item{saveDir}{The directory where the ptrSet object will be saved as .RData. 
If NULL, nothing will be saved}

\item{...}{may be used to pass parameters to the processFileTemporal function}

\item{timeLimit}{index time of the expiration limits and background. 
Should be provided by \code{\link[ptairMS]{timeLimits}} function}

\item{knots}{numeric vector corresponding to the knot values, which used for 
the two dimensional regression for each file. Should be provided 
by \code{\link[ptairMS]{defineKnots}} function}

\item{mzPrimaryIon}{the exact mass of the primary ion isotope}
}
\value{
an S4 ptrSet object, that contains the input ptrSet completed with the 
peakLists.
}
\description{
The \code{detectPeak} function detects peaks on the average total spectrum 
around nominal masses, for all files present in ptrSet which have not already 
been processed. The temporal evolution of each peak is then evaluated by using
a two-dimensional penalized spline regression. Finally, 
the expiration points (if defined in the ptrSet) are averaged, 
and a t-test is performed between expiration and ambient 
air. The peakList can be accessed with the \code{\link[ptairMS]{getPeakList}} 
function, which returns the information about the detected peaks in each file 
as a list of ExpressionSet objects.The peak detection steps within each file 
are as follows: \cr
for each nominal mass:
\itemize{
\item correction of the calibration drift
\item peak detection on the average spectrum
\item estimation of temporal evolution 
\item t-test between expiration and ambient air
}
}
\examples{

## For ptrRaw object
library(ptairData)
filePath <- system.file('extdata/exhaledAir/ind1', 'ind1-1.h5', 
package = 'ptairData')
raw <- readRaw(filePath,mzCalibRef=c(21.022,59.049),calib=TRUE)
timeLimit<-timeLimits(raw,fracMaxTIC=0.7)
knots<-defineKnots(object = raw,timeLimit=timeLimit)
raw <- detectPeak(raw, timeLimit=timeLimit, mzNominal = c(21,59),
smoothPenalty=0,knots=knots,resolutionRange=c(2000,5000,8000))

## For a ptrSet object
library(ptairData)
directory <- system.file("extdata/exhaledAir",  package = "ptairData")
exhaledPtrset<-createPtrSet(dir=directory,setName="exhaledPtrset",
mzCalibRef=c(21.022,59.049),
fracMaxTIC=0.9,saveDir= NULL)
exhaledPtrset  <- detectPeak(exhaledPtrset)
peakListEset<-getPeakList(exhaledPtrset)
Biobase::fData(peakListEset[[1]])
Biobase::exprs(peakListEset[[1]])
}
\references{
Muller et al 2014, Holzinger et al 2015, Marx and Eilers 1992
}
