#' Sample distance matrix 
#'
#'@description
#' Draws a simple hierarchical clustered heatmap to observe
#'sample distance.
#'
#'
#' @param data data.frame; originally generated by [mobileRNA::RNAimport()]
#' @param colours character; the colour palette. Default is 
#'` grDevices::colorRampPalette(RColorBrewer::brewer.pal(9, "GnBu"))(100)`
#' 
#' @param vst logical; to undertake variance stabilizing transformation. 
#' By default, the function uses a regularized log transformation on the data 
#' set, however, this will not suit all experimental designs.
#'
#'@param cellwidth numeric; individual cell width in points. Default is 40. 
#'
#'@param cellheight numeric; individual cell height in points. Default is 40. 
#'
#' @return A blue/green scale heatmap illustrating the sample distance.
#'
#' @details In special conditions, regularized log transformation will not suit
#' the experimental design. For example, an experimental design without
#' replicates. In this instance, it is preferable to change the default setting
#' and switch to a variance stabilizing transformation method (\code{vst=TRUE}).
#'
#'
#' @examples
#'
#' data("sRNA_data")
#' p1 <- plotSampleDistance(sRNA_data)
#'
#' @export
#' @importFrom DESeq2 rlog
#' @importFrom stats dist
#' @importFrom pheatmap pheatmap
#' @importFrom dplyr select
#' @importFrom grDevices colorRampPalette
#' @importFrom RColorBrewer brewer.pal
#' @importFrom grid unit
#' @importFrom grid gpar
plotSampleDistance <- function(data, 
                               colours = grDevices::colorRampPalette(rev(RColorBrewer::brewer.pal(9,"GnBu")))(255), 
                               vst = FALSE, cellheight = 40, cellwidth = 40){
  if (base::missing(data) || !base::inherits(data, c("data.frame"))) {
    stop("data must be an object of class data.frame containing raw count data")
  }

  data <- as.matrix(data %>% dplyr::select(tidyselect::starts_with("Count")))

  if(vst == TRUE){
    message("Transforming the count data with a variance stabilizing transformation...")
    rld <- DESeq2::varianceStabilizingTransformation(data, blind = TRUE)
    # log transform the data.
  } else
    if(vst == FALSE) {
      message("Transforming the count data to the log2 scale... \n")
      rld <- DESeq2::rlog(data, blind = TRUE) # log transform the data.
    }
  message("---Calculating distance matrix.")
  sample_names <- colnames(data)
  sample_names <- sub("Count_", "", sample_names)
  distance <- stats::dist(t(rld))
  distance_matrix <- as.matrix(distance)
  rownames(distance_matrix) <- paste(sample_names)
  colnames(distance_matrix) <- paste(sample_names)
  message("---Generating sample distance plot.")
  plot <- pheatmap::pheatmap(distance_matrix,
                             clustering_distance_rows = distance,
                             clustering_distance_cols = distance,
                             col = colours, 
                             angle_col = 45,
                             silent = TRUE,
                             cellheight=cellheight,
                             cellwidth=cellwidth) 
  
  plot$gtable$grobs[[4]]$gp <- grid::gpar(fontface = "bold") 
  plot$gtable$grobs[[5]]$gp <- grid::gpar(fontface = "bold")
  
  plot$gtable$widths[4] <-  plot$gtable$widths[4] + grid::unit(15, "bigpts")
  

  return(plot)
}






