#' Subset sRNA data based on dicercall size 
#'
#' @description Subset the existing dataframe to contain only the desired sRNA
#' class(s) based on the consensus dicercall determination and statistical 
#' significance. 
#' 
#' @details
#' See [mobileRNA::RNAdicercall()] for information on defining the consensus 
#' sRNA dicercall class for each cluster. The function allows the choice to 
#' filtered the data by statistical significance based on differential expression 
#' analysis, see [mobileRNA::RNAdifferentialAnalysis()]. Set 
#' \code{statistical=TRUE} to filtered by statistical significance (p-adjusted). 
#' 
#' 
#' It is important to consider what point in your analysis the data is subset 
#' as it will drastically reduce your sample size, altering the output of 
#' statistical analyse. 
#'
#' @param data data.frame; generated by [mobileRNA::RNAimport()], containing 
#' the additional information generated by [mobileRNA::RNAdicercall()] which 
#' defines the consensus sRNA class for each sRNA dicer-derived clusters
#'
#' @param class numeric; sRNA dicercall class(es) to select. Based on size in 
#' nucleotides. 
#' 
#' @param statistical logical; filter and select statistically significant sRNA. 
#' If \code{statistical=TRUE}, sRNA clusters are selected based on 
#' p-adjusted alpha threshold.
#' 
#' @param alpha numeric; user-defined numeric value to represent the adjusted 
#' p-value threshold to define statistical significance. Defaults setting 
#' `alpha=0.05`. 
#'
#' @return A data frame containing only sRNA clusters defined with a specific 
#' sRNA dicer-derived consensus size.
#' 
#' @examples
#' # load data 
#'  data("sRNA_data")
#'
#' # define consensus sRNA classes.
#' conditions <- c("heterograft_1", "heterograft_2", "heterograft_3")
#'
#' # Run function to define sRNA class for each cluster.
#' sRNA_data_dicercall <- RNAdicercall(data = sRNA_data,
#'                                   conditions = conditions,
#'                                   tidy=TRUE)
#'
#' # Subset data for  24-nt sRNAs
#' sRNA_24 <- RNAsubset(sRNA_data_dicercall, class = 24)
#'
#'
#'# Subset data for 21/22-nt sRNAs
#'sRNA_2122 <- RNAsubset(sRNA_data_dicercall, class = c(21, 22))
#'
#' @export
#' @importFrom dplyr %>%
#' @importFrom dplyr filter

RNAsubset <- function(data, class,  statistical=FALSE, alpha = 0.05){
  if (!base::inherits(data, c("matrix", "data.frame", "DataFrame"))) {
    stop("data must be an object of class matrix, data.frame, DataFrame")
  }
  if (base::missing(class) || !base::inherits(class, "numeric")) {
    stop("Please specify a numeric vector stating the class of sRNAs to select")
  }
    x <- data %>% dplyr::filter(DicerConsensus %in% class)
    if(statistical){
      x<- x %>%
        dplyr::filter(padj < alpha)
    }
    return(x)
  }

