% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VisualiseIndividualGenes.R
\name{VisualiseIndividualGenes}
\alias{VisualiseIndividualGenes}
\title{VisualiseIndividualGenes: Wrapper for Visualising Individual Genes in Gene Sets}
\usage{
VisualiseIndividualGenes(type, data, genes, metadata = NULL, ...)
}
\arguments{
\item{type}{Character. Specifies the type of plot to generate. Must be one of:
\itemize{
\item \code{"violin"}: Violin plots of individual gene expression by group
\item \code{"correlation"}: Correlation heatmap of selected genes
\item \code{"expression"}: Expression heatmap of selected genes
\item \code{"roc"}: ROC plots for classification performance of individual genes
\item \code{"auc"}: AUC plots for classification performance of individual genes
\item \code{"rocauc"}: Combined ROC and AUC plots
\item \code{"cohend"}: Effect size (Cohen's D) heatmap
\item \code{"pca"}: PCA plot of selected genes
}}

\item{data}{Required. Expression data matrix or data frame, with samples as
rows and genes as columns.}

\item{genes}{Required. Character vector of gene names to include in the
visualisation.}

\item{metadata}{Optional. Data frame with sample metadata, required for some
plot types (e.g., violin, roc, cohend).}

\item{...}{Additional arguments passed to the specific plotting function.}
}
\value{
The output of the specific plotting function called, usually a ggplot
or ComplexHeatmap object,
often wrapped in a list with additional data.
}
\description{
This wrapper function helps explore individual gene behavior within a gene
set used in \code{markeR}.
It dispatches to specific visualisation functions based on \code{plot_type},
supporting various plot types: heatmaps, violin plots, correlation analysis,
PCA, ROC/AUC, and effect size heatmaps.
}
\section{Additional required arguments (passed via \code{...}) per \code{plot_type}}{

\describe{
\item{violin}{Requires \code{GroupingVariable} (column name in
\code{metadata} for grouping).}
\item{roc, auc, rocauc}{Requires \code{condition_var} (metadata column
with condition labels) and \code{class} (positive class label).}
\item{cohend}{Requires \code{condition_var} and \code{class}
(same as roc).}
\item{correlation, expression, pca}{No additional mandatory arguments
required.}
}
}

\examples{
# Example data
set.seed(123)
expr_data <- matrix(rexp(1000, rate = 1), nrow = 50, ncol = 20)
rownames(expr_data) <- paste0("Gene", 1:50)
colnames(expr_data) <- paste0("Sample", 1:20)

sample_info <- data.frame(
  SampleID = colnames(expr_data),
  Condition = rep(c("A", "B"), each = 10),
  Diagnosis = rep(c("Disease", "Control"), each = 10),
  stringsAsFactors = FALSE
)
rownames(sample_info) <- sample_info$SampleID

selected_genes <- row.names(expr_data)[1:5]

# Violin plot
VisualiseIndividualGenes(
  type = "violin",
  data = expr_data,
  metadata = sample_info,
  genes = selected_genes,
  GroupingVariable = "Condition",
  nrow=1
)
VisualiseIndividualGenes(
  type = "correlation",
  data = expr_data,
  genes = selected_genes
)

# Expression heatmap
VisualiseIndividualGenes(
  type = "expression",
  data = expr_data,
  genes = selected_genes
)

# PCA plot
VisualiseIndividualGenes(
  type = "pca",
  data = expr_data,
  genes = selected_genes,
  metadata = sample_info,
  ColorVariable="Condition"

)

# ROC plot
VisualiseIndividualGenes(
  type = "roc",
  data = expr_data,
  metadata = sample_info,
  genes = selected_genes,
  condition_var = "Diagnosis",
  class = "Disease"

)

# AUC plot
VisualiseIndividualGenes(
  type = "auc",
  data = expr_data,
  metadata = sample_info,
  genes = selected_genes,
  condition_var = "Diagnosis",
  class = "Disease"
)

# ROC&AUC plot
VisualiseIndividualGenes(
  type = "rocauc",
  data = expr_data,
  metadata = sample_info,
  genes = selected_genes,
  condition_var = "Diagnosis",
  class = "Disease"
)

# Cohen's D plot
VisualiseIndividualGenes(
  type = "cohend",
  data = expr_data,
  metadata = sample_info,
  genes = selected_genes,
  condition_var = "Diagnosis",
  class = "Disease"
)



}
\seealso{
\code{\link{IndividualGenes_Violins}},
\code{\link{CorrelationHeatmap}},
\code{\link{ExpressionHeatmap}},
\code{\link{CohenD_IndividualGenes}},
\code{\link{plotPCA}},
\code{\link{ROCandAUCplot}}
}
