% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fuzzy.ttest.R
\name{fuzzy.ttest}
\alias{fuzzy.ttest}
\title{Function to compute the fuzzy Student t test based on weighted
mean and weighted variance}
\usage{
fuzzy.ttest(x, w1, w2, alternative=c("two.sided", "less", "greater"),
  check.w = TRUE, na.rm = FALSE)
}
\arguments{
\item{x}{an object containing the observed values.}

\item{w1}{a numerical vector of weights of the same length as x giving the weights
to use for elements of x in the first class.}

\item{w2}{a numerical vector of weights of the same length as x giving the weights to
use for elements of x in the second class.}

\item{alternative}{a character string specifying the alternative hypothesis, must be one
of "two.sided" (default), "greater" or "less". You can specify just the initial letter.}

\item{check.w}{TRUE if weights should be checked such that \verb{0 <= w <= 1} and \verb{w1[i] + w2[i]) < 1}
for 1 <= i <= length(x), FALSE otherwise. Beware that weights greater than one
may inflate over-optimistically resulting p-values, use with caution.}

\item{na.rm}{TRUE if missing values should be removed, FALSE otherwise.}
}
\value{
A numeric vector of six values that are the difference between the two weighted means,
the value of the t statistic, the sample size of class 1, the sample size of class 2,
the degree of freedom and the corresponding p-value.
}
\description{
This function allows for computing the weighted mean and weighted variance
of a vector of continuous values.
}
\details{
The weights w1 and w2 should represent the likelihood for each observation stored in
x to belong to the first and second class, respectively. Therefore the values contained
in w1 and w2 should lay in [0,1] and \verb{0 <= (w1[i] + w2[i]) <= 1} for i in {0,1,...,n} where
n is the length of x.
The Welch's version of the t test is implemented in this function, therefore assuming
unequal sample size and unequal variance. The sample size of the first and second class
are calculated as the sum(w1) and sum(w2), respectively.
}
\examples{
set.seed(54321)
# random generation of 50 normally distributed values for each of the two classes
xx <- c(rnorm(50), rnorm(50)+1)
# fuzzy membership to class 1
ww1 <- runif(50) + 0.3
ww1[ww1 > 1] <- 1
ww1 <- c(ww1, 1 - ww1)
# fuzzy membership to class 2
ww2 <- 1 - ww1
# Welch's t test weighted by fuzzy membership to class 1 and 2
wt <- fuzzy.ttest(x=xx, w1=ww1, w2=ww2)
print(wt)
# Not run:
# permutation test to compute the null distribution of the weighted t statistic
wt <- wt[2]
rands <- t(sapply(1:1000, function(x,y) { return(sample(1:y)) }, y=length(xx)))
randst <- apply(rands, 1, function(x, xx, ww1, ww2)
{ return(fuzzy.ttest(x=xx, w1=ww1[x], w2=ww2[x])[2]) }, xx=xx, ww1=ww1, ww2=ww2)
ifelse(wt < 0, sum(randst <= wt), sum(randst >= wt)) / length(randst)
# End(Not run)

}
\references{
http://en.wikipedia.org/wiki/T_test
}
\seealso{
\link[stats:weighted.mean]{stats::weighted.mean}
}
