#' initReportFilename
#'
#' @param object an easyreporting class object 
#' @param filenamepath the name of the report with or without the path
#' @param mainTitle the title of the report
#' @param author the name of the report author
#' @param documentType type of report final document. (html is default)
#'
#' @return an easyreporting class object 
#' @keywords internal
setGeneric (
    name="initReportFilename",
    def=function(object, filenamepath=NULL, title=NULL,
                author=NULL, optionList=NULL, bibfile=NULL, documentType=NULL,
                resources=NULL)
    {
        standardGeneric("initReportFilename")
    }
)
#' mkdSetGlobalOpts
#'
#' @param object an easyreporting class object
#' @param optionList a list of options
#'
#' @return an easyreporting class object
#' @keywords internal
setGeneric (
    name="mkdSetGlobalOpts",
    def=function(object, optionList=list()){standardGeneric("mkdSetGlobalOpts")}
)
#' mkdTitle
#' @description Inserts an rmarkdown title inside the report, useful for create
#' distinct sections inside the final resport.
#' @param object an easyreporting class object
#' @param title a string within the title.
#' @param level a numeric from 1 to 6 (default is 1).
#'
#' @return none
#' @export
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#'
#' mkdTitle(rd, "First Level Title")
#' mkdTitle(rd, "Sub-Title", level=2)
#' }
setGeneric (
    name="mkdTitle",
    def=function(object, title, level=1){standardGeneric("mkdTitle")}
)
#' getReportFilename
#' 
#' @description returns the filename with its path of the report generated by 
#' easyreporting
#'
#' @param object an easyreporting class object
#'
#' @return a string of report file name with path
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' (rep <- getReportFilename(rd))
#' }
setGeneric (
    name="getReportFilename",
    def=function(object){standardGeneric("getReportFilename")}
)
#' mkdGeneralMsg
#' @description It appends a general message to the report, useful for 
#' describing a code chunk or a part of the report.
#' Useful for adding natural language comments.
#' @param object an easyreporting class object
#' @param message the message to append to the report
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' mkdGeneralMsg(rd, message="Writing a paragraph to describe my code chunk")
#' }
setGeneric (
    name="mkdGeneralMsg",
    def=function(object, message){standardGeneric("mkdGeneralMsg")}
)

#' mkdGeneralTitledMsg
#' @description It appends a a titled section followed by a general message to 
#' the report. Useful for adding natural language comments.
#' @param object an easyreporting class object
#' @param title the title of the report section.
#' @param level the level (1 to 6) of the title (default is 1)
#' @param message the message to append to the report
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' mkdGeneralTitledMsg(rd, title="Generic SubTitle for this message", level=2,
#'             message="Writing a paragraph to describe my code chunk")
#' }
setGeneric (
    name="mkdGeneralTitledMsg",
    def=function(object, title=NULL, level=1, message)
        {standardGeneric("mkdGeneralTitledMsg")}
)


#' setOptionsList
#' @description set an optionList to the class, it automatically sets the 
#' options for the rmarkdown file. Are useful for the 
#' interpretation/execution of the code chunks.
#' 
#'
#' @param object an easyreporting class object
#' @param cacheFlag boolean for caching chunk data (default TRUE)
#' @param evalFlag boolean for evaluating the code chunk in the compiled version
#' (default TRUE)
#' @param echoFlag boolean for showing the code chunk (default TRUE)
#' @param warningFlag boolean for showing the chunk warnings (default FALSE)
#' @param showMessages boolean for showing the chunk warnings in compiled
#' version (default FALSE)
#' @param includeFlag boolean for including the code chunk in the compiled
#' version (default TRUE)
#' @param collapseFlag boolean for collapsing the code chunk in the compiled
#' version (default FALSE),
#' @param purlFlag boolean for extracting the code chunk as R code in a separate
#' R file (default TRUE),
#' @param errorFlag boolean for including the error generated by the code chunk 
#' in the compiled version (default TRUE),
#' @param messageFlag boolean for including the code chunk messages in the compiled
#' version (default TRUE),
#' @param highlightFlag boolean for highlinghtinh the code chunk in the compiled
#' version (default TRUE),
#' @param promptFlag boolean for including a ">" for the code chunk in the compiled
#' version (default FALSE),
#' @param stripWhiteFlag boolean for removing the white spaces at beginning/end 
#' of the code chunk in the compiled version (default TRUE),
#' @param tidyFlag boolean for creating a tidy code chunk in the compiled
#' version (default FALSE).
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#'
#' ## setting default option
#' setOptionsList(rd)
#'
#' ## modifying only some options
#' rd <- setOptionsList(rd, warningFlag=TRUE, 
#'         showMessages=TRUE, includeFlag=TRUE)
#' }
setGeneric (
    name="setOptionsList",
    def=function(object, cacheFlag=TRUE,
                 evalFlag=TRUE,
                 echoFlag=TRUE,
                 warningFlag=FALSE,
                 showMessages=FALSE,
                 includeFlag=TRUE,
                 collapseFlag=FALSE,
                 purlFlag=TRUE,
                 errorFlag=TRUE,
                 messageFlag=TRUE,
                 highlightFlag=TRUE,
                 promptFlag=FALSE,
                 stripWhiteFlag=TRUE,
                 tidyFlag=FALSE)
    {standardGeneric("setOptionsList")}
)
#' getOptionsList
#'
#' @description returns the optionList from the easyreporting class 
#' (see the makeOptionList function for more details).
#' @param object an easyreporting class object
#'
#' @return a list of options
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report", title="example_report",
#'                         author=c("It's me"))
#' optList <- getOptionsList(rd)
#' }
setGeneric (
    name="getOptionsList",
    def=function(object){standardGeneric("getOptionsList")}
)
#' compile
#' @description prints the sessionInfo and compiles the produced rmarkdown file
#' into an HTML report.
#' @param object an easyreporting class object
#'
#' @return none
#' @export
#' 
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report", title="example_report",
#'                         author=c("It's me"))
#' compile(rd)
#' }
setGeneric (
    name="compile",
    def=function(object){standardGeneric("compile")}
)
#' mkdVariableAssignment
#'
#' @description it includes a variable assignment in the report.
#' NB: a call to the "mkdCodeChunkSt" has to be done before using it.
#'
#' @param object an easyreporting class object
#'
#' @param variable.name a string indicating the name of the variabe to store in
#' the report. (This can be changed here, but further uses of the variable needs
#' to take into account the variable name change).
#' @param variable.object.name the name of the already existing variable. (This
#' cannot be canged.)
#' @param show a boolean indicating if to show the message before writing it
#' into the rmardown file.
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' ## leaving the default options to the code chunk
#' mkdCodeChunkSt(rd)
#' ## adding a variable assignement
#' variable <- 1
#' mkdVariableAssignment(rd, "variable", "variable", show=TRUE)
#' mkdCodeChunkEnd(rd)
#' }
setGeneric (
    name="mkdVariableAssignment",
    def=function(object, variable.name, variable.object.name, show)
    {
        standardGeneric("mkdVariableAssignment")
    }
)

#' mkdCodeChunkSt
#' @description it creates a code chunk start. A list of options and files to
#' source  for the chunk can optionally be passed to the function.
#' @param object an easyreporting class object
#' @param optionList a list of options
#' @param sourceFilesList a list of files that can be sourced inside the code
#' chunk.
#' @param isComplete a flag determining if the chunk is already a complete chunk
#'
#' @return none
#' @export
#'
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' ## no options
#' mkdCodeChunkSt(rd)
#' ## just leaving empty
#' mkdCodeChunkEnd(rd)
#'
#' ## setting options
#' optList <- makeOptionsList(includeFlag=TRUE)
#' mkdCodeChunkSt(rd, optionList=optList)
#' ## just leaving empty
#' mkdCodeChunkEnd(rd)
#' }
setGeneric (
    name="mkdCodeChunkSt",
    def=function(object, optionList=getOptionsList(object),
                 sourceFilesList=NULL, isComplete=FALSE)
    {standardGeneric("mkdCodeChunkSt")}
)
#' mkdSourceFiles
#' @description includes a list of source files inside the rmarkdown
#' @param object an easyreporting class object
#' @param ... a list of files to source
#'
#' @return none
#'
#' @keywords internal
setGeneric (
    name="mkdSourceFiles",
    def=function(object, ...){standardGeneric("mkdSourceFiles")}
)

#' mkdCodeChunkEnd
#' @description it creates a code chunk end. Always use it after a
#' mkdCodeChunkSt()
#' @param object an easyreporting class object
#' @return none
#'
#' @export
#'
#' @examples
#' \dontrun{
#'  rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' mkdCodeChunkSt(rd)
#' ## just leaving empty
#' mkdCodeChunkEnd(rd)
#' }
setGeneric (
    name="mkdCodeChunkEnd",
    def=function(object){standardGeneric("mkdCodeChunkEnd")}
)
#' mkdCodeChunkComplete
#' @description it creates a complete code chunk.
#' @param object an easyreporting class object
#' @param code a string or an expression generated with \link[base]{quote} 
#' containing a function call or the entire code chunk to trace.
#' @param optionList a list of options.
#' @param sourceFilesList a list of files to source.
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#'  rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' mkdCodeChunkComplete(rd, code="a <- 1\n b <- 2\n c <- a+b\n print(c)")
#' }
setGeneric (
    name="mkdCodeChunkComplete",
    def=function(object, code,
                optionList=getOptionsList(object),
                sourceFilesList=NULL){standardGeneric("mkdCodeChunkComplete")}
)
#' mkdCodeChunkCommented
#' @description it creates a complete code chunk, adding a natural language
#' comment before of it.
#' @param object an easyreporting class object
#' @param comment a string with the natural language comment for the chunk.
#' @param code a string within the code.
#' @param optionList a list of options (default is the class options).
#' @param sourceFilesList a optional list of files to source inside the chunk.
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' optList <- makeOptionsList(includeFlag=TRUE, cacheFlag=TRUE)
#' mkdCodeChunkCommented(rd,
#'                 comment="This is the comment of the following code chunk",
#'                 code="a <- 1\n b <- 2\n (c <- a+b)\n", optionList=optList,
#'                 sourceFilesList=NULL)
#' }
setGeneric (
    name="mkdCodeChunkCommented",
    def=function(object, comment=NULL, code,
                optionList=getOptionsList(object),
                sourceFilesList=NULL){standardGeneric("mkdCodeChunkCommented")}
)
#' mkdCodeChunkCommented
#' @description it creates a complete code chunk, adding a natural language
#' comment before of it.
#' @param object an easyreporting class object
#' @param title the title to assign to the code chunk section
#' @param level the level of the title (default is 1)
#' @param comment a string with the natural language comment for the chunk.
#' @param code a string within the code.
#' @param optionList a list of options (default is the class options).
#' @param sourceFilesList a optional list of files to source inside the chunk.
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' optList <- makeOptionsList(includeFlag=TRUE, cacheFlag=TRUE)
#' mkdCodeChunkCommented(rd,
#'                 comment="This is the comment of the following code chunk",
#'                 code="a <- 1\n b <- 2\n (c <- a+b)\n", optionList=optList,
#'                 sourceFilesList=NULL)
#' }

setGeneric (
    name="mkdCodeChunkTitledCommented",
    def=function(object, title=NULL, level=1, comment=NULL, code,
                optionList=getOptionsList(object),
                sourceFilesList=NULL){
                standardGeneric("mkdCodeChunkTitledCommented")}
)

#' setBibliography
#' @description add a bibfile name to the object that will be reflected into the
#' report as a bibliography section
#' @param object an easyreporting class object
#' @param bibfile a path string with the name of the bib file
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' TBD
#' }
setGeneric (
    name="setBibliography",
    def=function(object, bibfile=NULL){
        standardGeneric("setBibliography")
    }
)


#' getBibliography
#' @description returns the bibfile name attached to the object
#' @param object an easyreporting class object
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' TBD
#' }
setGeneric (
    name="getBibliography",
    def=function(object){
        standardGeneric("getBibliography")
    })


#' addResource
#'
#' @param object an easyreporting class instance
#' @param source a string indicating the reference (i.e. "GEO")
#' @param reference a string indicanting the reference of the source (i.e. "GSE60231)
#' @param description a natural language description 
#'
#' @return an easyreporting class instance
#' @export
#'
#' @examples
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#' rd <- addResource(rd, source="GEO", "GSE60231", "Transcriptome of BMDC to different antigen delivery systems")
#' \dontrun{
#' compile(rd)
#' }
setGeneric("addResource", function(object, source, reference, description)
{
    standardGeneric("addResource")
})

