##' RnaSeqParam constructor
##' 
##' This constructs a \code{\linkS4class{RnaSeqParam}} object, that combines
##' all the necessary parameters for the analysis of RNA-Seq data. As much as
##' possible, these parameters are determined automa-gi/ti-cally. It describes 
##' three sets of parameters:
##' \itemize{
##'   \item parameters describing the annotation
##'   \item parameters describing the BAM files, \emph{i.e.} the type of
##'   sequencing that was conducted.
##'   \item parameters describing how the counting should be done.
##' }
##' The first two are provided through sepcific objects: \code{\linkS4class{AnnotParam}} and
##' \code{\linkS4class{BamParam}} respectively. The third one is a set 
##' constituted of:
##' \itemize{
##'   \item countBy: the feature per which the counts should be summarized (
##'   exon, transcript or gene. A forth possibility - feature - can be used to
##'   define arbitrary genomic loci)
##'   \item precision: the precision at which the counts should be performed:
##'   bp or reads. bp used to be the default in the \code{easyRNASeq} package,
##'   whereas now reads is, following the Bioconductor main stream development.
##' }
##' The default parameters for the \code{\linkS4class{BamParam}} parameter are
##' derived from the currently most common RNA-Seq experimental use-case: 
##' strand-specific paired-end Illumina sequencing. See the respective manual
##' pages of \code{\linkS4class{AnnotParam}} and
##' \code{\linkS4class{BamParam}} for more details.
##' 
##' @aliases RnaSeqParam RnaSeqParam,ANY-method
##' @name easyRNASeq RnaSeqParam constructor
##' @rdname easyRNASeq-RnaSeqParam
##' @param annotParam An object derived from class \code{AnnotParam}.
##' @param bamParam An object derived from class \code{BamParam}.
##' @param countBy TODO
##' @param precision A character value, either 'read' or 'bp' that
##' defines the precision at which counting is done, either per read 
##' or per covered bp. 'read' is the default.
##' @examples
## create the necessary AnnotParam
##' annotParam <- AnnotParam(
##'                 datasource=system.file(
##'                                 "extdata",
##'                                 "Dmel-mRNA-exon-r5.52.gff3",
##'                                 package="RnaSeqTutorial"))
##' 
##' ## create the RnaSeqParam
##' rsp <- RnaSeqParam(annotParam=annotParam)
##' 
##' ## change some defaults
##' RnaSeqParam(countBy="features",annotParam=annotParam)
##' RnaSeqParam(bamParam=BamParam(stranded=TRUE,yieldSize=1L),annotParam=annotParam)
##' 
##' 
setMethod(f="RnaSeqParam",
          signature="ANY",
          definition=function(
            annotParam=AnnotParam(),
            bamParam=BamParam(),
            countBy=c("exons","features","genes","transcripts"),
            precision=c("read","bp")){

            precision <- match.arg(precision)
            countBy <- match.arg(countBy)
              
            ## what was that for anyway????
#             countBy <- switch(precision,
#                               "read"="transcripts",
#                               "bp"="exons"
#                    )
#             
            new("RnaSeqParam",
                annotParam=annotParam,
                bamParam=bamParam,
                countBy=countBy,
                precision=precision)
          })
