% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/retrieve_easier_score.R
\name{retrieve_easier_score}
\alias{retrieve_easier_score}
\title{Retrieve easier scores of immune response}
\usage{
retrieve_easier_score(
  predictions_immune_response = NULL,
  TMB_values = NULL,
  easier_with_TMB = c("weighted_average", "penalized_score"),
  weight_penalty,
  verbose = TRUE
)
}
\arguments{
\item{predictions_immune_response}{list containing the predictions
for each quantitative descriptor and for each task. This is the
output from \code{predict_immune_response}.}

\item{TMB_values}{numeric vector containing patients' tumor
mutational burden (TMB) values.}

\item{easier_with_TMB}{character string indicating which approach
should be used to integrate easier with TMB: "weighted_average"
(default) and "penalized_score".}

\item{weight_penalty}{integer value from 0 to 1, which is used to
define the weight or penalty for combining easier and TMB scores
based on a weighted average or penalized score, in order to derive
a score of patient's likelihood of immune response. The default
value is 0.5.}

\item{verbose}{logical flag indicating whether to display messages
about the process.}
}
\value{
A data.frame with samples in rows and easier scores in columns.
}
\description{
Calculates easier score and if applicable, both weighted average and
penalized score based on the combination of easier score and TMB.
}
\examples{
# using a SummarizedExperiment object
library(SummarizedExperiment)
# Using example exemplary dataset (Mariathasan et al., Nature, 2018)
# from easierData. Original processed data is available from
# IMvigor210CoreBiologies package.
library("easierData")

dataset_mariathasan <- easierData::get_Mariathasan2018_PDL1_treatment()
RNA_tpm <- assays(dataset_mariathasan)[["tpm"]]
cancer_type <- metadata(dataset_mariathasan)[["cancertype"]]

# Select a subset of patients to reduce vignette building time.
pat_subset <- c(
  "SAM76a431ba6ce1", "SAMd3bd67996035", "SAMd3601288319e",
  "SAMba1a34b5a060", "SAM18a4dabbc557"
)
RNA_tpm <- RNA_tpm[, colnames(RNA_tpm) \%in\% pat_subset]

# Computation of TF activity (Garcia-Alonso et al., Genome Res, 2019)
tf_activities <- compute_TF_activity(
  RNA_tpm = RNA_tpm
)

# Predict patients' immune response
predictions <- predict_immune_response(
  tfs = tf_activities,
  cancer_type = cancer_type,
  verbose = TRUE
)

# retrieve clinical response
patient_ICBresponse <- colData(dataset_mariathasan)[["BOR"]]
names(patient_ICBresponse) <- colData(dataset_mariathasan)[["pat_id"]]

# retrieve TMB
TMB <- colData(dataset_mariathasan)[["TMB"]]
names(TMB) <- colData(dataset_mariathasan)[["pat_id"]]

patient_ICBresponse <- patient_ICBresponse[names(patient_ICBresponse) \%in\% pat_subset]
TMB <- TMB[names(TMB) \%in\% pat_subset]

easier_derived_scores <- retrieve_easier_score(
  predictions_immune_response = predictions,
  TMB_values = TMB,
  easier_with_TMB = c("weighted_average", "penalized_score"),
  weight_penalty = 0.5
)
\donttest{

RNA_counts <- assays(dataset_mariathasan)[["counts"]]
RNA_counts <- RNA_counts[, colnames(RNA_counts) \%in\% pat_subset]

# Computation of cell fractions  (Finotello et al., Genome Med, 2019)
cell_fractions <- compute_cell_fractions(RNA_tpm = RNA_tpm)

# Computation of pathway scores (Holland et al., BBAGRM, 2019;
# Schubert et al., Nat Commun, 2018)
pathway_activities <- compute_pathway_activity(
  RNA_counts = RNA_counts,
  remove_sig_genes_immune_response = TRUE
)

# Computation of ligand-receptor pair weights
lrpair_weights <- compute_LR_pairs(
  RNA_tpm = RNA_tpm,
  cancer_type = "pancan"
)

# Computation of cell-cell interaction scores
ccpair_scores <- compute_CC_pairs(
  lrpairs = lrpair_weights,
  cancer_type = "pancan"
)

# Predict patients' immune response
predictions <- predict_immune_response(
  pathways = pathway_activities,
  immunecells = cell_fractions,
  tfs = tf_activities,
  lrpairs = lrpair_weights,
  ccpairs = ccpair_scores,
  cancer_type = cancer_type,
  verbose = TRUE
)

easier_derived_scores <- retrieve_easier_score(
  predictions_immune_response = predictions,
  TMB_values = TMB,
  easier_with_TMB = c("weighted_average", "penalized_score"),
  weight_penalty = 0.5
)
}
}
