% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/markovProbability.R
\name{markovProbability}
\alias{markovProbability}
\title{Markov Chain Construction and Probability Calculation}
\usage{
markovProbability(
  milo,
  diffusionmap,
  terminal_state = NULL,
  root_cell,
  knn = 30L,
  diffusiontime = NULL,
  pseudotime_key = "pseudotime",
  scale_components = TRUE,
  num_waypoints = 500,
  n_eigs = NULL,
  verbose = TRUE,
  use_RANN = TRUE
)
}
\arguments{
\item{milo}{A \code{Milo} or \code{SingleCellExperiment} object. This object should
have pseudotime stored in \code{colData}, which will be used to calculate
probabilities. If pseudotime is available in \code{milo}, it takes precedence over
the value provided through the \code{diffusiontime} parameter.}

\item{diffusionmap}{A \code{DiffusionMap} object corresponding to the \code{milo}
object. Used for Markov chain construction.}

\item{terminal_state}{Integer. The index of the terminal state in the Markov
chain.}

\item{root_cell}{Integer. The index of the root state in the Markov chain.}

\item{knn}{Integer. The number of nearest neighbors for graph construction.
Default is \code{30L}.}

\item{diffusiontime}{Numeric vector. If pseudotime is not stored in \code{milo},
this parameter can be used to provide pseudotime values to the function.}

\item{pseudotime_key}{Character. The name of the column in \code{colData} that
contains the inferred pseudotime.}

\item{scale_components}{Logical. If \code{TRUE}, the components will be scaled
before constructing the Markov chain. Default is \code{FALSE}.}

\item{num_waypoints}{Integer. The number of waypoints to sample when
constructing the Markov chain. Default is \code{500L}.}

\item{n_eigs}{integer, default is NULL. Number of eigen vectors to use.
\itemize{
\item If is not specified, the number of eigen vectors will be determined using
the eigen gap.
}}

\item{verbose}{Logical. If \code{TRUE}, print progress. Default is \code{TRUE}.}

\item{use_RANN}{parameter to make user choose
whether to use RANN to construct Markov chain,
or keep using bluster}
}
\value{
milo or SinglCellExperiment object with pseudotime, probabilities in
its colData
}
\description{
This function preprocesses data, constructs a Markov chain, and calculates
transition probabilities based on pseudotime information.
}
\examples{
data(sce_vdj)
# downsample to first 2000 cells
sce_vdj <- sce_vdj[, 1:2000]
sce_vdj <- setupVdjPseudobulk(sce_vdj,
    already.productive = FALSE,
    allowed_chain_status = c("Single pair", "Extra pair")
)
# Build Milo Object
set.seed(100)
milo_object <- miloR::Milo(sce_vdj)
milo_object <- miloR::buildGraph(milo_object,
    k = 50, d = 20,
    reduced.dim = "X_scvi"
)
milo_object <- miloR::makeNhoods(milo_object,
    reduced_dims = "X_scvi",
    d = 20
)

# Construct Pseudobulked VDJ Feature Space
pb.milo <- vdjPseudobulk(milo_object, col_to_take = "anno_lvl_2_final_clean")
pb.milo <- scater::runPCA(pb.milo, assay.type = "Feature_space")

# Define root and branch tips
pca <- t(as.matrix(SingleCellExperiment::reducedDim(pb.milo, type = "PCA")))
branch.tips <- c(which.min(pca[, 2]), which.max(pca[, 2]))
names(branch.tips) <- c("CD8+T", "CD4+T")
root <- which.min(pca[, 1])

# Construct Diffusion Map
dm <- destiny::DiffusionMap(t(pca), n_pcs = 10, n_eigs = 5)
dif.pse <- destiny::DPT(dm, tips = c(root, branch.tips), w_width = 0.1)

# Markov Chain Construction
pb.milo <- markovProbability(
    milo = pb.milo,
    diffusionmap = dm,
    diffusiontime = dif.pse[[paste0("DPT", root)]],
    terminal_state = branch.tips,
    root_cell = root,
    pseudotime_key = "pseudotime"
)
}
