% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/conversion.R
\name{conversion}
\alias{conversion}
\alias{spectraVariableMapping,character-method}
\alias{spectraVariableMapping,missing-method}
\alias{setSpectraVariableMapping}
\alias{defaultSpectraVariableMapping}
\alias{r_to_py.Spectra}
\alias{rspec_to_pyspec}
\alias{pyspec_to_rspec}
\title{Converting between R and Python MS data structures}
\usage{
\S4method{spectraVariableMapping}{character}(object, x = character(), ...)

\S4method{spectraVariableMapping}{missing}(object, ...)

setSpectraVariableMapping(x)

defaultSpectraVariableMapping()

\method{r_to_py}{Spectra}(x, convert = FALSE)

rspec_to_pyspec(
  x,
  mapping = spectraVariableMapping(),
  pythonLibrary = c("matchms", "spectrum_utils")
)

pyspec_to_rspec(
  x,
  mapping = spectraVariableMapping(),
  pythonLibrary = c("matchms", "spectrum_utils")
)
}
\arguments{
\item{object}{For \code{spectraVariableMapping()}: not used.}

\item{x}{\code{Spectra} object.}

\item{...}{For \code{spectraVariableMapping()}: not used.}

\item{convert}{Boolean; should Python objects be automatically converted to
their R equivalent? Defaults to \code{FALSE}.}

\item{mapping}{named \code{character()} vector defining which spectra
variables/metadata should be translated between R and Python and how
they should be renamed. Defaults to \code{spectraVariableMapping()}.}

\item{pythonLibrary}{For \code{rspec_to_pyspec()} and \code{pyspec_to_rspec()}:
\code{character(1)} defining the Python library to which (or from which)
data structures the data should be converted.
Possible options are \code{"matchms"} or \code{"spectrum_utils"} with \code{"matchms"}
being the default.}
}
\value{
For \code{r_to_py.Spectra()} and \code{rspec_to_pyspec()}: Python list of
MS data structures, either \code{matchms.Spectrum} or
\code{spectrum_utils.spectrum.MsmsSpectrum} objects. For \code{pyspec_to_rspec()}:
\code{\link[Spectra:Spectra]{Spectra::Spectra()}} with the MS data of all \code{matchms.Spectrum} objects
in the submitted \code{list}.
}
\description{
The \code{rspec_to_pyspec()} and \code{pyspec_to_rspec()} functions allow to convert
(translate) MS data structures between R and Python. At present the
R \code{\link[Spectra:Spectra]{Spectra::Spectra()}} objects can be either translated into a list of
\href{https://github.com/matchms/matchms}{matchms} Python \code{matchms.Spectrum}
objects or
\href{https://github.com/bittremieux-lab/spectrum_utils}{spectrum_utils} Python
\code{spectrum_utils.spectrum.MsmsSpectrum} objects.
For better integration with the \emph{reticulate} R package also a
\code{r_to_py.Spectra()} method is available.

The mapping of spectra variables (in R) to (Python) spectra metadata can
be configured and defined with the \code{setSpectraVariableMapping()} and
\code{spectraVariableMapping()}. These get and set the \emph{global} (system wide)
setting and are thus also used by the \code{r_to_py()} method.

Properties for translation to the MS data objects of the different Python
libraries are:
\itemize{
\item \emph{matchms}: the \code{matchms.Spectrum} objects support arbitrary metadata, so
any spectra variable can be translated and stored in these objects.
\item \emph{spectrum_utils}: the \code{spectrum_utils.spectrum.MsmsSpectrum} object
supports metadata variables \emph{identifier} (\code{character}), \emph{precursor_mz}
(\code{numeric}), \emph{precursor_charge} (\code{integer}) and optionally also
\emph{retention_time} (\code{numeric}).
}

See the indivudual function's documentation for more details.

Function to convert R Spectra objects into a Python list of matchms Spectrum
objects using the \code{reticulate} package.
}
\section{Translation of MS data objects}{


MS data structures can be translated between R and Python using the
\code{rspec_to_pyspec()} and \code{pyspec_to_rspec()} functions, or with the
\code{r_to_py()} method.
\itemize{
\item \code{rspec_to_pyspec()} translates an R \code{\link[Spectra:Spectra]{Spectra::Spectra()}} object into a
list of Python MS data objects, which can be, depending on parameter
\code{pythonLibrary}, \code{matchms.Spectrum} objects (for
\code{pythonLibrary = "matchms"}, the default) or
\code{spectrum_utils.spectrum.MsmsSpectrum} objects (for
\code{pythonLibrary = "spectrum_utils"}). Parameter \code{mapping} allows to specify
which spectra variables from the \code{Spectra} object \code{x} should be converted
in addition to the peaks data (m/z and intensity values). It defaults to
\code{mapping = spectraVariableMapping()} (See the respective help below for
more information on the variable mapping). While being fast, this function
first loads all peaks and spectra data into memory before translating to
Python data structures. A less memory intense operation could be to call
this function in a loop to only load parts of the data at a time into
memory.
\item \code{pyspec_to_rspec()} translates a single, or a list of \code{matchms.Spectrum}
objects (with parameter \code{pythonLibrary = "matchms"}, the default) or a
list of \code{spectrum_utils.spectrum.MsmsSpectrum} objects (with parameter
\code{pythonLibrary = "spectrum_utils"}) to a \code{\link[Spectra:Spectra]{Spectra::Spectra()}} object.
Parameter \code{mapping} allows to specify the metadata variables that
should be translated and mapped in addition to the peaks data. The
library used to represent the MS data in Python needs to be specified with
parameter \code{pythonLibrary}.
\item \code{r_to_py.Spectra()} is equivalent to
\code{rspec_to_pyspec(pythonLibrary = "matchms")}. The spectra
variables that should be converted can be configures with
\code{setSpectraVariableMapping()} (see documentation below).
}
}

\section{Mapping of spectra variables (metadata)}{


Metadata for MS spectra are represented and stored as \emph{spectra variables}
in the R \code{\link[Spectra:Spectra]{Spectra::Spectra()}} objects. Also Python MS data structures
store such metadata along with the mass peak data. While spectra metadata
is thus supported by data structures in both programming languages,
different names and naming conventions are used. The
\code{spectraVariableMapping()} and \code{setSpectraVariableMapping()} functions allow
to define how the names of spectra metadata (spectra variables) should be
translated between R and Python. To support also the different naming
conventions used by the Python libraries \emph{matchms} and \emph{spectrum_utils},
\code{spectraVariableMapping()} defines different mapping schemes for these,
using by default the mapping for \emph{matchms}. Note also that \emph{spectrum_utils}
supports only few selected metadata/spectra variables, so any additional
spectra variables defined by the mapping will be ignored.
The \code{r_to_py()} and \code{py_to_r()} functions will use the selected naming
scheme to name the spectra variables accordingly. Also, only
spectra metadata/variables in \code{spectraVariableMapping()} will be translated.
The initial mapping is based on this
\href{https://github.com/matchms/matchms/blob/master/matchms/data/known_key_conversions.csv}{definition in matchms}.
\itemize{
\item \code{defaultSpectraVariableMapping()}: returns the \emph{default} mapping between
spectra variables and Python metadata names for the \emph{matchms} library.
\item \code{spectraVariableMapping()}: returns the currently defined spectra variable
mapping as a named character vector, with names
representing the names of the spectra variables in R and elements the
respective names of the spectra metadata in Python.
Use \code{\link[Spectra:spectraData]{Spectra::spectraVariables()}} on the \code{Spectra} object that should
be converted with \code{r_to_py()} to list all available spectra variables.
\code{r_to_py()} and \code{py_to_r()} for MS data structures will use this default
mapping.
Calling \code{spectraVariableMapping()} defining also the Python library (e.g.,
\code{spectraVariableMapping("matchms")} or
\code{spectraVariableMapping("spectrum_utils")}) will return the variable
mapping for the specified Python library. Optional parameter \code{x} allows
to specify a (potentially names) character vector with the names of the
spectra variables that should in addition be included in the mapping.
\item \code{setSpectraVariableMapping()}: sets/replaces the currently defined mapping
of spectra variable names to Python metadata names. Setting
\code{setSpectraVariableMapping(character())} will only convert the mass peaks
data (m/z and intensity values) but no spectra metadata.
}
}

\examples{

## Import a MGF file as a `Spectra` object
library(MsBackendMgf)
library(SpectriPy)
s <- Spectra(
    system.file("extdata", "mgf", "spectra2.mgf", package = "SpectriPy"),
    source = MsBackendMgf())
s

#########################
## Conversion R to Python

## A `Spectra` can be translated to a `list` of `matchms.Spectrum` objects
## using either the `r_to_py()` method or the `rspec_to_pyspec()` function:
s_py <- r_to_py(s)
s_py

## The `s_py` can now be used like any other Python variable within the R
## *reticulate* framework. Below we extract the m/z values of the first
## spectrum
s_py[0]$mz

## Extracting that information from the `Spectra` object in R
s[1]$mz

## The `spectraVariableMapping()` defines which spectra variables (metadata)
## should be translated between R and Python:
spectraVariableMapping()

## The names of that character vector represent the names of the spectra
## variables in R, the elements the name of the metadata variable in Python.
## Below we list the available metadata information from the first
## Spectrum in Python
s_py[0]$metadata

## `setSpectraVariableMapping()` allows to replace the default mapping
## of variables. Below we e.g. add a new spectra variable to the `Spectra`
## object.
s$new_col <- 1:4

## To translate that variable to Python we need to include it to the
## `spectraVariableMapping()`. Below we define to translate only the
## precursor m/z and the new spectra variable to Python. Be aware that
## `setSpectraVariableMapping()` **globally** sets the default for any
## spectra variable mapping between R and Python. Thus, any subsequent
## calls mapping calls will use the same mapping. It is suggested to
## eventually *restore* the default mapping again after the call or
## use the `rspec_to_pyspec()` function instead, that allows to configure
## the mapping using a parameter `mapping`.
setSpectraVariableMapping(
    c(precursorMz = "precursor_mz", new_col = "new_col"))
s_py <- r_to_py(s)

s_py[0]$metadata

## Restoring the global spectra variable mapping configuration to
## the default mapping:
setSpectraVariableMapping(defaultSpectraVariableMapping())

## As an alternative to the `r_to_py()` we can use the `rspec_to_pyspec()`
## function and provide a custom mapping using the `mapping` parameter:
s_py <- rspec_to_pyspec(
    s, mapping = c(precursorMz = "precursor_mz", new_col = "new_col"))

## Convert to MS data objects from the spectrum_utils Python library
s_py2 <- rspec_to_pyspec(
    s, mapping = spectraVariableMapping("spectrum_utils"),
    pythonLibrary = "spectrum_utils")

## Convert the data back to R
pyspec_to_rspec(s_py2, pythonLibrary = "spectrum_utils")

#########################
## Conversion Python to R

## A `list` of `matchms.Spectrum` objects in Python can be translated into
## the corresponding MS data structure in R (i.e. a `Spectra`) object using
## the `pyspec_to_rspec()` function:
res <- pyspec_to_rspec(s_py)
res

## All spectra from Python are thus converted into a single `Spectra` object.

## Or providing a custom variable mapping:
res <- pyspec_to_rspec(
    s_py, mapping = c(precursorMz = "precursor_mz", new_col = "new_col"))
res$new_col
}
\author{
Michael Witting, Johannes Rainer, Wout Bittremieux, Thomas Naake
}
