#' Convert GFF to a binary occurrence matrix
#'
#' This function takes a GFF data frame and converts it into a binary matrix,
#' indicating the presence (1) or absence (0) of a feature in a sequence.
#'
#' @param input A data frame containing GFF data, typically generated by
#'   `rtracklayer::import.gff` and converted to a data frame. It must have
#'   'type', 'start', 'end', and 'seqnames' columns.
#' @return A matrix, where values are binary: `1` indicates the presence of a
#'   feature, and `0` indicates its absence.
#' @examples
#' gff_file <- system.file("extdata/out_Hb_gff.txt", package = "PMScanR")
#' if (nzchar(gff_file)) {
#'   gff_data <- as.data.frame(rtracklayer::import.gff(gff_file))
#'   motif_matrix <- gff2matrix(gff_data)
#'   # print(head(motif_matrix))
#' }
#' @importFrom reshape2 dcast
#' @export
gff2matrix <- function(input) {
    if (nrow(input) == 0) {
        return(matrix(nrow = 0, ncol = 0))
    }
    input$matchID <-
        paste0(input$type, ":", input$start, "-", input$end)
    at <- input[, c("seqnames", "matchID")]
    at <- unique(at) #should be uniqe right ?
    at$exist <- 1
    df_at <-
        reshape2::dcast(at,
                        matchID ~ seqnames,
                        value.var = "exist",
                        fill = 0)
    rownames(df_at) <- df_at$matchID
    df_at <- df_at[, -1, drop = FALSE]
    
    return(as.matrix(df_at))
}
