#' Convert numbers to more readable strings
#' 
#' Format raw numbers to more readable strings. For example, 1000 will be
#' converted to "1K". Supported suffixes are "K", "M", and "B".
#' 
#' @param x A number.
#' @param decimal_digits Number of decimal digits to round to.
#' 
#' @returns A character string of the formatted number. \code{NA} is returned as
#' "NA".
#' 
#' @examples
#' print(MotifPeeker:::pretty_number(134999))
#' 
#' @keywords internal
pretty_number <- function(x, decimal_digits = 2) {
    if (is.na(x)) return("NA")
    if (x < 1e3) {
        return(as.character(round(x, decimal_digits)))
    } else if (x < 1e6) {
        return(paste0(round(x / 1e3, decimal_digits), "K"))
    } else if (x < 1e9) {
        return(paste0(round(x / 1e6, decimal_digits), "M"))
    } else {
        return(paste0(round(x / 1e9, decimal_digits), "B"))
    }
}

#' Pipe operator
#'
#' See \code{magrittr::\link[magrittr:pipe]{\%>\%}} for details.
#' 
#' Generated by \code{\link[usethis]{use_pipe}}. Modified to import from
#' \code{dplyr} instead of \code{magrittr}.
#'
#' @name %>%
#' @rdname pipe
#' @keywords internal
#' @export
#' @importFrom dplyr %>%
#' @usage lhs \%>\% rhs
#' @param lhs A value or the magrittr placeholder.
#' @param rhs A function call using the magrittr semantics.
#' 
#' @examples
#' seq_len(10) %>% sum
#' 
#' @returns The result of calling `rhs(lhs)`.
NULL

#' Format exp_type
#' 
#' Format input exp_type to look pretty.
#' 
#' @param exp_type A character depicting the type of experiment.
#' Supported experimental types are:
#' \itemize{
#'   \item \code{chipseq}: ChIP-seq data
#'   \item \code{tipseq}: TIP-seq data
#'   \item \code{cuttag}: CUT&Tag data
#'   \item \code{cutrun}: CUT&Run data
#'   \item \code{other}: Other experiment type data
#'   \item \code{unknown}: Unknown experiment type data
#' }
#' Any item not mentioned above will be returned as-is.
#' 
#' @returns A character vector of formatted exp_type.
#' 
#' @examples
#' MotifPeeker:::format_exptype("chipseq")
#' 
#' @keywords internal
format_exptype <- function(exp_type) {
    if (is.na(exp_type)) {
        exp_type <- "unknown"
    }
    
    exp_type <- tolower(exp_type)
    exp_types <- c(
        "tipseq" = "TIP-Seq",
        "chipseq" = "ChIP-Seq",
        "cutrun" = "CUT&RUN",
        "cuttag" = "CUT&Tag",
        "other" = "Other",
        "unknown" = "Unknown"
    )
    
    if (!exp_type %in% names(exp_types)) {
        return(exp_type)
    }
    
    exp_type <- exp_types[[exp_type]]
    return(exp_type)
}

#' Print messages 
#' 
#' Conditionally print messages.
#'  Allows developers to easily control verbosity of functions, 
#'  and meet Bioconductor requirements that dictate the message 
#'  must first be stored to a variable before passing to \link[base]{message}. 
#'  
#' 
#' @param v Whether to print messages or not.
#' 
#' @returns Null
#' 
#' @keywords internal 
messager <- function(..., v = Sys.getenv("VERBOSE") != "FALSE") {
    msg <- paste(...)
    if (v) try({message(msg)})
}

#' Check, add and access files in cache
#' 
#' Query local BiocFileCache to get cached version of a file and add them if
#' they do not exist.
#' 
#' @param url A character string specifying the URL of the file to check for.
#' @inheritParams MotifPeeker
#' 
#' @importFrom BiocFileCache BiocFileCache bfcinfo bfcrpath
#' 
#' @returns A character string specifying the path to the cached file.
#' 
#' @keywords internal
use_cache <- function(url, verbose = FALSE) {
    bfc <- BiocFileCache::BiocFileCache(ask = FALSE)
    cached_files <- BiocFileCache::bfcinfo(bfc)$rname
    
    if (!url %in% cached_files) {
        messager("Adding file to cache: ", url, v = verbose)
    }
    return(BiocFileCache::bfcrpath(bfc, url))
}

#' Check for duplicates
#' 
#' Checks for duplicated items in a vector or list and throw an error if found.
#' 
#' @param x A vector or list.
#' 
#' @returns Null
#' 
#' @keywords internal
check_duplicates <- function(x) {
    stp_msg <- paste("Duplicated items found in the label list. Please input",
                    "unique experiment and motif labels.")
    if (any(duplicated(x))) {
        stop(stp_msg)
    }
}

#' Check attached dependency
#' 
#' Stop execution if a package is not attached.
#' 
#' @param pkg a character string of the package name
#' @param fatal a logical value indicating whether to stop execution if the
#' package is not attached.
#' @param custom_msg a custom message to display if the package is not attached.
#' 
#' @returns TRUE if the package is available or else FALSE if
#' \code{fatal = FALSE}.
#' 
#' @keywords internal
check_dep <- function(pkg, fatal = TRUE, custom_msg = NULL){
    if (is.null(custom_msg)) {
        custom_msg <- paste("Package", shQuote(pkg), "is required to run this",
                            "function.")
    }
    if (!requireNamespace(pkg, quietly = TRUE)) {
        if (fatal) {
            stop(custom_msg)
        } else {
            warning(custom_msg)
            return(FALSE)
        }
    }
    return(TRUE)
}

#' Stop if MEME suite is not installed
#' 
#' @param continue Continue code execution if MEME suite is not installed.
#' @inheritParams memes::runFimo
#' 
#' @importFrom memes meme_is_installed
#' 
#' @returns Null
#' 
#' @seealso \code{\link[memes]{check_meme_install}}
#' 
#' @keywords internal
confirm_meme_install <- function(meme_path = NULL, continue = FALSE) {
    msg <- paste(
        "Cannot find MEME suite installation. If installed, try setting the",
        "path", shQuote("MEME_BIN"), "environment varaible, or use the",
        shQuote("meme_path"), "parameter in the MotifPeeker function call.",
        "\nFor more information, see the memes pacakge documention-",
        "\nhttps://github.com/snystrom/memes#detecting-the-meme-suite"
    )
    
    if (!memes::meme_is_installed(meme_path)) {
        if (continue) {
            messager(msg)
            return(FALSE)
        } else {
            stop(msg)
        }
    }
    return(TRUE)
}

#' Generate a random string
#'
#' @param length The length of the random string to generate.
#'
#' @returns A random string of the specified length.
#' 
#' @keywords internal
random_string <- function(length) {
    char_base <- c(0:9, letters)
    random_chars <- sample(char_base, length, replace = TRUE)
    return(paste(random_chars, collapse = ""))
}

#' Apply \code{\link[base]{normalizePath}} to a list of paths
#' 
#' @param path_list A list of paths.
#' 
#' @returns A list of normalised paths or the input as is if contents are not
#' a character.
#' 
#' @keywords internal
normalise_paths <- function(path_list) {
    if (all(is.null(path_list))) return(path_list)
    
    ## Convert objects to list if not already
    if (!is.list(path_list) &&(!is.vector(path_list) || length(path_list) == 1))
        path_list <- list(path_list)
    
    ## Return input as is if not character
    if (!all(vapply(path_list, is.character, logical(1)))) return(path_list)
    
    ## Normalise paths
    lapply(path_list, function(path) {
        normalizePath(path)
    })
}
