#' @title classifyMut
#'
#' @param maf  Maf or MafList object generated by \code{\link{readMaf}} function.
#' Classify SSNVs/Indels into Shared/P-shared/Private, Clonal/Subclonl
#' or Shared-Clonal/P-shared-Clonal/Private-Clonal/Shared-Subclonal/P-shared-SubClonal/Private-SubClonal 
#' @param patient.id  Select the specific patients. Default NULL, all patients are included
#' @param class  The class which would be represented. Default: "SP" (Shared pattern: Public/Shared/Private),
#' other options: "CS" (Clonal status: Clonal/Subclonl) and "SPCS".
#' @param classByTumor  Logical (Default: FALSE). Classify mutations based on "Tumor_ID".
#' @param ... Other options passed to \code{\link{subMaf}}
#' @return A data.frame with classification of mutations for each patient 
#' 
#' @examples
#' maf.File <- system.file("extdata/", "CRC_HZ.maf", package = "MesKit")
#' clin.File <- system.file("extdata/", "CRC_HZ.clin.txt", package = "MesKit")
#' ccf.File <- system.file("extdata/", "CRC_HZ.ccf.tsv", package = "MesKit")
#' maf <- readMaf(mafFile=maf.File, clinicalFile = clin.File, ccfFile=ccf.File, refBuild="hg19")
#' classifyMut(maf, class = "SP")
#' @export classifyMut


classifyMut <- function(
     maf,     
     patient.id = NULL,
     class = "SP",
     classByTumor = FALSE,
     ...) {
    
    ## check input data
    maf_input <- maf_data <- subMaf(maf,
                                   patient.id = patient.id,
                                   mafObj = TRUE,
                                   ...)
    
    result <- list()
    for(m in maf_input){
        maf_data <- getMafData(m)
        patient <- getMafPatient(m)
        if(nrow(maf_data) == 0){
            message("Warning: there was no mutation in ", patient, " after filtering.")
            next
        }
        mut.class <- do.classify(
            maf_data, 
            class = class,
            classByTumor = classByTumor
        ) %>% 
            dplyr::mutate(Mut_ID = paste(
                .data$Hugo_Symbol, 
                .data$Chromosome,
                .data$Start_Position,
                .data$Reference_Allele, 
                .data$Tumor_Seq_Allele2,
                sep = ":"
            )) %>%
            dplyr::select(
                "Patient_ID",
                "Tumor_Sample_Barcode",
                "Mut_ID",
                "Mutation_Type")
        
        result[[patient]] <- mut.class
        
    }
    
    if(length(result) > 1){
        return(result)
    }else if(length(result) == 0){
        return(NA)
    }else{
        return(result[[1]])
    }

    return(mut.class)

}
