% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DataClasses.R, R/functions-MSpectra.R,
%   R/methods-MSpectra.R
\name{MSpectra}
\alias{MSpectra}
\alias{MSpectra-class}
\alias{show,MSpectra-method}
\alias{coerce,MSpectra,list-method}
\alias{coerce,MSpectra,MSnExp-method}
\alias{mz,MSpectra-method}
\alias{intensity,MSpectra-method}
\alias{rtime,MSpectra-method}
\alias{precursorMz,MSpectra-method}
\alias{precursorCharge,MSpectra-method}
\alias{precScanNum,MSpectra-method}
\alias{precursorIntensity,MSpectra-method}
\alias{acquisitionNum,MSpectra-method}
\alias{scanIndex,MSpectra-method}
\alias{peaksCount,MSpectra,ANY-method}
\alias{msLevel,MSpectra-method}
\alias{tic,MSpectra-method}
\alias{ionCount,MSpectra-method}
\alias{collisionEnergy,MSpectra-method}
\alias{fromFile,MSpectra-method}
\alias{polarity,MSpectra-method}
\alias{smoothed,MSpectra-method}
\alias{isEmpty,MSpectra-method}
\alias{centroided,MSpectra-method}
\alias{isCentroided,MSpectra-method}
\alias{writeMgfData,MSpectra-method}
\alias{clean,MSpectra-method}
\alias{removePeaks,MSpectra-method}
\alias{filterMz,MSpectra-method}
\alias{pickPeaks,MSpectra-method}
\alias{smooth,MSpectra-method}
\alias{filterMsLevel,MSpectra-method}
\title{List of Spectrum objects along with annotations}
\usage{
MSpectra(..., elementMetadata = NULL)

\S4method{mz}{MSpectra}(object)

\S4method{intensity}{MSpectra}(object)

\S4method{rtime}{MSpectra}(object)

\S4method{precursorMz}{MSpectra}(object)

\S4method{precursorCharge}{MSpectra}(object)

\S4method{precScanNum}{MSpectra}(object)

\S4method{precursorIntensity}{MSpectra}(object)

\S4method{acquisitionNum}{MSpectra}(object)

\S4method{scanIndex}{MSpectra}(object)

\S4method{peaksCount}{MSpectra,ANY}(object)

\S4method{msLevel}{MSpectra}(object)

\S4method{tic}{MSpectra}(object)

\S4method{ionCount}{MSpectra}(object)

\S4method{collisionEnergy}{MSpectra}(object)

\S4method{fromFile}{MSpectra}(object)

\S4method{polarity}{MSpectra}(object)

\S4method{smoothed}{MSpectra}(object)

\S4method{isEmpty}{MSpectra}(x)

\S4method{centroided}{MSpectra}(object)

\S4method{isCentroided}{MSpectra}(object)

\S4method{writeMgfData}{MSpectra}(object, con = "spectra.mgf", COM = NULL, TITLE = NULL)

\S4method{clean}{MSpectra}(object, all = FALSE, msLevel. = msLevel., ...)

\S4method{removePeaks}{MSpectra}(object, t, msLevel., ...)

\S4method{filterMz}{MSpectra}(object, mz, msLevel., ...)

\S4method{pickPeaks}{MSpectra}(
  object,
  halfWindowSize = 3L,
  method = c("MAD", "SuperSmoother"),
  SNR = 0L,
  refineMz = c("none", "kNeighbors", "kNeighbours", "descendPeak"),
  msLevel. = unique(msLevel(object)),
  ...
)

\S4method{smooth}{MSpectra}(
  x,
  method = c("SavitzkyGolay", "MovingAverage"),
  halfWindowSize = 2L,
  ...
)

\S4method{filterMsLevel}{MSpectra}(object, msLevel.)
}
\arguments{
\item{...}{For \code{MSpectra}: \linkS4class{Spectrum} object(s) or a \code{list} of
\linkS4class{Spectrum} objects.
For all other methods optional arguments passed along.}

\item{elementMetadata}{For \code{MSpectra}: \link{DataFrame} with optional information
that should be added as metadata information (\code{mcols}) to the object.
The number of rows has to match the number of \linkS4class{Spectrum} objects,
each row is expected to represent additional metadata information for
one spectrum.}

\item{object}{For all functions: a \code{MSpectra} object.}

\item{x}{For all functions: a \code{MSpectra} object.}

\item{con}{For \code{writeMgfData}: \code{character(1)} defining the file name of
the MGF file.}

\item{COM}{For \code{writeMgfData}: optional \code{character(1)} providing a comment
to be added to the file.}

\item{TITLE}{For \code{writeMgfData}: optional \code{character(1)} defining the title
for the MGF file.}

\item{all}{For \code{clean}: if \code{FALSE} original 0-intensity values are retained
around peaks.}

\item{msLevel.}{For \code{clean}, \code{removePeaks}, \code{filterMz}, \code{pickPeaks}:
optionally specify the MS level(s) of the spectra on which the operation
should be performed.
For \code{filterMsLevels}: MS level(s) to which the \code{MSpectra} should be reduced.}

\item{t}{For \code{removePeaks}: \code{numeric(1)} specifying the threshold below
which intensities are set to 0.}

\item{mz}{For \code{filterMz}: \code{numeric(2)} defining the lower and upper m/z
for the filter. See \code{\link[=filterMz]{filterMz()}} for details.}

\item{halfWindowSize}{For \code{pickPeaks} and \code{smooth}: see \code{\link[=pickPeaks]{pickPeaks()}}
and \code{\link[=smooth]{smooth()}} for details.}

\item{method}{For \code{pickPeaks} and \code{smooth}: see \code{\link[=pickPeaks]{pickPeaks()}} and \code{\link[=smooth]{smooth()}}
for details.}

\item{SNR}{For \code{pickPeaks}: see \code{\link[=pickPeaks]{pickPeaks()}} for details.}

\item{refineMz}{For \code{pickPeaks}: see \code{\link[=pickPeaks]{pickPeaks()}} for details.}
}
\description{
\code{MSpectra} (Mass Spectra) objects allow to collect one or more
\linkS4class{Spectrum} object(s) (\linkS4class{Spectrum1} or \linkS4class{Spectrum2}) in
a \code{list}-like structure with the possibility to add arbitrary annotations
to each individual \code{Spectrum} object. These can be accessed/set with
the \code{\link[=mcols]{mcols()}} method.

\code{MSpectra} objects can be created with the \code{MSpectra} function.

Functions to access the individual spectra's attributes are available
(listed below).

\code{writeMgfData} exports a \code{MSpectra} object to a file in MGF format. All
metadata columns present in \code{mcols} are exported as additional fields with
the capitalized column names used as field names (see examples below).
}
\details{
\code{MSpectra} inherits all methods from the \link{SimpleList} class of the
\code{S4Vectors} package. This includes \code{lapply} and other data manipulation
and subsetting operations.
}
\note{
Note that the \href{https://bioconductor.org/packages/Spectra}{Spectra} package
provides a more robust and efficient infrastructure for mass spectrometry
data handling and analysis. So, wherever possible, the newer \emph{Spectra}
package should be used instead of the \emph{MSnbase}.

For backward compatibility, it is however possible to convert between the
\code{MSpectra} and the newer \code{Spectra} objects:
\itemize{
\item A \code{Spectra} object can be coerced to a \code{MSpectra} using
\code{as(sps, "MSpectra")} where \code{sps} is a \code{Spectra} object.
\item The \code{\link[=extractSpectraData]{extractSpectraData()}} function can be used to extract the data from
a \code{MSpectra} as a \code{DataFrame}, which can then be used to create a
\code{Spectra} object.
}
}
\section{Constructor}{


New \link{MSpectra} can be created with the \code{MSpectra(...)} function
where \code{...} can either be a single \linkS4class{Spectrum} object or a \code{list} of
\code{Spectrum} objects (\linkS4class{Spectrum1} and/or \linkS4class{Spectrum2}).
}

\section{Accessing spectrum attributes}{


These methods allow to access the attributes and values of the individual
\code{Spectrum} (\linkS4class{Spectrum1} or \linkS4class{Spectrum2}) objects within the list.
\itemize{
\item \code{mz} return the m/z values of each spectrum as a \code{list} of \code{numeric}
vectors.
\item \code{intensity} return the intensity values of each spectrum as a \code{list} of
\code{numeric} vectors.
\item \code{rtime} return the retention time of each spectrum as a \code{numeric} vector
with length equal to the length of \code{object}.
\item \code{precursorMz}, \code{precursorCharge}, \code{precursorIntensity}, \code{precScanNum}
return precursor m/z values, charge, intensity and scan number for each
spectrum as a \code{numeric} (or \code{integer}) vector with length equal to the
length of \code{object}. Note that for \linkS4class{Spectrum1} objects \code{NA} will be
returned.
\item \code{acquisitionNum} and \code{scanIndex} return the acquisition number of each
spectrum and its scan index as an \code{integer} vector with the same length
than \code{object}.
\item \code{ionCount} and \code{tic} return the ion count and total ion current of each
spectrum.
\item \code{peaksCount} returns the number of peaks for each spectrum as a \code{integer}
vector.
\item \code{msLevel} returns the MS level of each spectrum.
\item \code{collisionEnergy} returns the collision energy for each spectrum or \code{NA}
for \linkS4class{Spectrum1} objects.
\item \code{polarity} returns the spectra's polarity.
\item \code{fromFile} returns the index from the (e.g. mzML) file the spectra where
from. This applies only for spectra read using the \code{\link[=readMSData]{readMSData()}} function.
\item \code{smoothed} whether spectra have been smoothed (i.e. processed with the
\code{\link[=smooth]{smooth()}} method. Returns a \code{logical} of length equal to the
number of spectra.
\item \code{isEmpty} returns \code{TRUE} for spectra without peak data.
\item \code{centroided}, \code{isCentroided} returns for each spectrum whether it contains
\emph{centroided} data. While \code{centroided} returns the internal attribute of
each spectrum, \code{isCentroided} tries to guess whether spectra are
centroided from the actual peak data.
}
}

\section{Data manipulation methods}{

\itemize{
\item \code{clean} \emph{cleans} each spectrum. See \code{\link[=clean]{clean()}} for more details.
\item \code{pickPeaks} performs peak picking to generate centroided spectra. See
\code{\link[=pickPeaks]{pickPeaks()}} for more details.
\item \code{removePeaks} removes peaks lower than a threshold \code{t}. See
\code{\link[=removePeaks]{removePeaks()}} for more details.
\item \code{smooth} \emph{smooths} spectra. See \code{\link[=smooth]{smooth()}} for more details.
}
}

\section{Filtering and subsetting}{

\itemize{
\item \code{[} can be used to subset the \code{MSpectra} object.
\item \code{filterMsLevel} filters \code{MSpectra} to retain only spectra from certain MS
level(s).
\item \code{filterMz} filters the spectra by the specified \code{mz} range. See
\code{\link[=filterMz]{filterMz()}} for details.
}
}

\examples{

## Create from Spectrum objects
sp1 <- new("Spectrum1", mz = c(1, 2, 4), intensity = c(4, 5, 2))
sp2 <- new("Spectrum2", mz = c(1, 2, 3, 4), intensity = c(5, 3, 2, 5),
    precursorMz = 2)

spl <- MSpectra(sp1, sp2)
spl
spl[[1]]

## Add also metadata columns
mcols(spl)$id <- c("a", "b")
mcols(spl)

## Create a MSpectra with metadata
spl <- MSpectra(sp1, sp2, elementMetadata = DataFrame(id = c("a", "b")))

mcols(spl)
mcols(spl)$id

## Extract the mz values for the individual spectra
mz(spl)

## Extract the intensity values for the individual spectra
intensity(spl)

## Extract the retention time values for the individual spectra
rtime(spl)

## Extract the precursor m/z of each spectrum.
precursorMz(spl)

## Extract the precursor charge of each spectrum.
precursorCharge(spl)

## Extract the precursor scan number for each spectrum.
precScanNum(spl)

## Extract the precursor intensity of each spectrum.
precursorIntensity(spl)

## Extract the acquisition number of each spectrum.
acquisitionNum(spl)

## Extract the scan index of each spectrum.
scanIndex(spl)

## Get the number of peaks per spectrum.
peaksCount(spl)

## Get the MS level of each spectrum.
msLevel(spl)

## Get the total ion current for each spectrum.
tic(spl)

## Get the total ion current for each spectrum.
ionCount(spl)

## Extract the collision energy for each spectrum.
collisionEnergy(spl)

## Extract the file index for each spectrum.
fromFile(spl)

## Get the polarity for each spectrum.
polarity(spl)

## Whether spectra are smoothed (i.e. processed with the `smooth`
## function).
smoothed(spl)

## Are spectra empty (i.e. contain no peak data)?
isEmpty(spl)

## Do the spectra contain centroided data?
centroided(spl)

## Do the spectra contain centroided data? Whether spectra are centroided
## is estimated from the peak data.
isCentroided(spl)

## Export the spectrum list to a MGF file. Values in metadata columns are
## exported as additional field for each spectrum.
tmpf <- tempfile()
writeMgfData(spl, tmpf)

## Evaluate the written output. The ID of each spectrum (defined in the
## "id" metadata column) is exported as field "ID".
readLines(tmpf)

## Set mcols to NULL to avoid export of additional data fields.
mcols(spl) <- NULL
file.remove(tmpf)

writeMgfData(spl, tmpf)
readLines(tmpf)

## Filter the object by MS level
filterMsLevel(spl, msLevel. = 1)
}
\author{
Johannes Rainer
}
