\name{essR}
\alias{essR}

\title{
	Global optimization algorithm for MINLPs based on Scatter Search
}

\description{
	  essR attempts to solve problems of the form:

	  \eqn{min f(x,p_1,p_2,...,p_n)}

subject to:

	 \eqn{c_e=0}\cr
	 \eqn{c_L \le c(x) \le c_U}\cr
	\eqn{x_L \le x \le x_U}\cr


%	\deqn{ min F(x)  subject to:
%	x\
%		ceq(x) = 0 (equality constraints)\cr
%		c_L <= c(x) <= c_U (inequality constraints)\cr
%		x_L <= x <= x_U (bounds on the decision variables)
}

\usage{
	essR(problem, opts = list(maxeval = NULL, maxtime = NULL), ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
 	\item{problem}{
    		List containing problem definition.
	}
  	\item{opts}{ 
		List containing options (if set as opts <- numeric(0) default options will be loaded).  
  	}

  	\item{...}{Additional variables passed to the objective function}
}

\details{
		Problem definition:

 		problem$f:	Name of the file containing the objective function (String).\cr
		problem$x_L: Lower bounds of decision variables (vector).\cr
		problem$x_U: Upper bounds of decision variables (vector).\cr
		problem$x_0: Initial point(s) (optional; vector or matrix).\cr
		problem$f_0: Function values of initial point(s) (optional). These values MUST correspond to feasible points.\cr

	NOTE: The dimension of f_0 and x_0 may be different. For example, if we want to introduce 5 initial points
 	but we only know the values for 3 of them, x_0 would have 5 rows whereas f_0 would have only 3  elements.
	In this example, it is mandatory that the first 3 rows of x_0 correspond to the values of f_0.\cr

	Fill the following fields if your problem has non-linear constraints:

       		problem$neq: Number of equality constraints (Integer; do not define it if there are no equality constraints).\cr
       		problem$c_L: Lower bounds of nonlinear inequality constraints (vector).\cr
       		problem$c_U: Upper bounds of nonlinear inequality constraints (vector).\cr
       		problem$int_var: Number of integer variables (Integer).\cr
       		problem$bin_var: Number of binary variables  (Integer).\cr
       		problem$vtr: Objective function value to be reached (optional).\cr

	User options:

		opts$maxeval: Maximum number of function evaluations  (Default 1000).\cr
		opts$maxtime: Maximum CPU time in seconds (Default 60).\cr
		opts$iterprint: Print each iteration on screen: 0-Deactivated; 1-Activated (Default 1).\cr
		opts$plot: Plots convergence curves: 0-Deactivated; 1-Plot curves on line; 2-Plot final results (Default 0).\cr
		opts$weight: Weight that multiplies the penalty term added to the objective function in constrained problems (Default 1e6).\cr
		opts$log_var: Indexes of the variables which will be used to generate diverse solutions in different orders of magnitude (vector).\cr
		opts$tolc: Maximum absolute violation of the constraints (Default 1e-5).\cr
		opts$prob_bound: Probability (0-1) of biasing the search towards the bounds (Default 0.5).\cr
		opts$inter_save: Saves results in a mat file in intermediate iterations. Useful for very long runs (Binary; Default = 0).\cr
	
	Global options:

          	opts$dim_refset: Number of elements in Refset (Integer; automatically calculated).\cr
           	opts$ndiverse: Number of solutions generated by the diversificator (Default 10*nvar).\cr
		opts$combination: Type of combination of Refset elements (Default 1);1: hyper-rectangles; 2: linear combinations.\cr

	Local options:

           	opts$local_solver: Choose local solver 0: Local search deactivated (Default), "NM", "BFGS", "CG", "LBFGSB","SA","SOLNP".\cr
           	opts$local_tol: Level of tolerance in local search.\cr
           	opts$local_iterprint: Print each iteration of local solver on screen (Binary; default = 0).\cr
           	opts$local_n1: Number of iterations before applying local search for the 1st time (Default 1).\cr
           	opts$local_n2: Minimum number of iterations in the global phase between 2 local calls (Default 10).\cr
           	opts$local_balance: Balances between quality (=0) and diversity (=1) for choosing initial points for the local search (default 0.5).\cr
           	opts$local_finish: Applies local search to the best solution found once the optimization if finished (same values as opts.local.solver).\cr
           	opts$local_bestx: When activated (i.e. =1) only applies local search to the best solution found to date,ignoring filters (Default=0).\cr
}

\value{
     \item{fbest}{Best objective function value found after the optimization}
     \item{xbest}{Vector providing the best function value}
     \item{cpu_time}{Time in seconds consumed in the optimization}
     \item{f}{Vector containing the best objective function value after each iteration}
     \item{x}{Matrix containing the best vector after each iteration}
     \item{time}{Vector containing the cpu time consumed after each iteration}
     \item{neval}{Vector containing the number of function evaluations after each iteration}
     \item{numeval}{Number of function evaluations}
     \item{ local_solutions}{Local solutions found by the local solver (in rows)}
     \item{local_solutions_values}{Function values of the local solutions}
     \item{end_crit}{ Criterion to finish the optimization:
	\tabular{ll}{
                                         1 \tab Maximal number of function evaluations achieved\cr
                                         2 \tab Maximum allowed CPU Time achieved\cr
                                         3 \tab Value to reach achieved\cr
	  }
      }
}
\references{

If you use essR and publish the results, please cite the following papers:

 Egea, J.A., Maria, R., Banga, J.R. (2010) An evolutionary method for
 complex-process optimization. Computers & Operations Research 37(2):
 315-324.
 
 Egea, J.A., Balsa-Canto, E., Garcia, M.S.G., Banga, J.R. (2009) Dynamic
 optimization of nonlinear processes with an enhanced scatter search
 method. Industrial & Engineering Chemistry Research 49(9): 4388-4401. 
}

\author{
	Jose Egea
}

\note{
     R code of the eSS optimization code from: Process Engineering Group IIM-CSIC.
   
   Constraint functions, if applicable, must be declared in the same script as the objective function as a second output argument, e.g.:
	\verb{
 	 myfunction <- function(x){
		calculate fx - scalar containing the objective function value
		calculate gx - vector (or empty) containing the constraints values
		return(list(fx,gx))
	}
	}
}

\examples{
#1 Unconstrained problem

ex1 <- function(x){
	y<-4*x[1]*x[1]-2.1*x[1]^4+1/3*x[1]^6+x[1]*x[2]-4*x[2]*x[2]+4*x[2]^4;
	return(y)
}

#global optimum
#x*=[0.0898, -0.7127];
# or    
#x*=[-0.0898, 0.7127];
#
#f(x*)= -1.03163;

#========================= PROBLEM SPECIFICATIONS ===========================
problem<-list(f="ex1",x_L=rep(-1,2),x_U=rep(1,2))
opts<-list(maxeval=500, ndiverse=10, dim_refset=4, local_solver="solnp", local_n2=1)
#========================= END OF PROBLEM SPECIFICATIONS =====================

Results<-essR(problem,opts);




#2 Constrained problem

ex2<-function(x){
	F=-x[1]-x[2];
	g<-rep(0,2);
	g[1]<-x[2]-2*x[1]^4+8*x[1]^3-8*x[1]^2;
	g[2]<-x[2]-4*x[1]^4+32*x[1]^3-88*x[1]^2+96*x[1];
	return(list(F=F,g=g))
}

# global optimum
#x*=[2.32952, 3.17849];    
#f(x*)=-5.50801

#========================= PROBLEM SPECIFICATIONS ===========================
problem<-list(f="ex2",x_L=rep(0,2),x_U=c(3,4), c_L=rep(-Inf,2), c_U=c(2,36))
opts<-list(maxeval=750, local_solver="solnp", local_n2=1)
#========================= END OF PROBLEM SPECIFICATIONS =====================

Results<-essR(problem,opts);




#3 Constrained problem with equality constraints

ex3<-function(x,k1,k2,k3,k4){
	f=-x[4];
#Equality constraints
	g<-rep(0,5);
	g[1]=x[4]-x[3]+x[2]-x[1]+k4*x[4]*x[6];
	g[2]=x[1]-1+k1*x[1]*x[5];
	g[3]=x[2]-x[1]+k2*x[2]*x[6];
	g[4]=x[3]+x[1]-1+k3*x[3]*x[5];

#Inequality constraint
	g[5]=x[5]^0.5+x[6]^0.5;
	return(list(f=f,g=g));
}

#global optimum
#x*=[0.77152
# 	0.516994
# 	0.204189
# 	0.388811
# 	3.0355
# 	5.0973];
# 
#f(x*)= -0.388811;

#========================= PROBLEM SPECIFICATIONS ===========================
problem<-list(f="ex3",x_L=rep(0,6),x_U=c(rep(1,4),16,16), neq=4, c_L=-Inf, c_U=4)
opts<-list(maxtime=7, local_solver="solnp", local_n2=10)
#========================= END OF PROBLEM SPECIFICATIONS =====================

k1=0.09755988;
k3=0.0391908;
k2=0.99*k1;
k4=0.9*k3;
Results<-essR(problem,opts,k1,k2,k3,k4);




#4 Mixed integer problem

ex4<-function(x){
	F = x[2]^2 + x[3]^2 + 2.0*x[1]^2 + x[4]^2 - 5.0*x[2] - 5.0*x[3] - 21.0*x[1] + 7.0*x[4];
	g<-rep(0,3);
	g[1] = x[2]^2 + x[3]^2 + x[1]^2 + x[4]^2 + x[2] - x[3] + x[1] - x[4];
	g[2] = x[2]^2 + 2.0*x[3]^2 + x[1]^2 + 2.0*x[4]^2 - x[2] - x[4];
	g[3] = 2.0*x[2]^2 + x[3]^2 + x[1]^2 + 2.0*x[2] - x[3] - x[4];
	return(list(F=F, g=g));
}

# global optimum
#x*=[2.23607, 0, 1, 0];
#f(x*)=-40.9575;

#========================= PROBLEM SPECIFICATIONS ===========================
problem<-list(f="ex4", x_L=rep(0,4), x_U=rep(10,4), x_0=c(3,4,5,1),int_var=3, c_L=rep(-Inf,3), c_U=c(8,10,5))
opts<-list(maxtime=2)
#========================= END OF PROBLEM SPECIFICATIONS =====================

Results<-essR(problem,opts);
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{scatter}
\keyword{search}% __ONLY ONE__ keyword per line
\keyword{optimization}
