#'
#' @title MCMC Chain Execution for HiCPotts Simulations
#'
#' @description
#' The run_metropolis_MCMC_betas function serves as a high-level wrapper to execute the MCMC sampling process on Hi-C interaction data.
#' It leverages the run_metropolis_MCMC_betas function to perform Bayesian inference, including updating model parameters (betas, gamma, size, and theta)
#' and mixture component assignments. It can also run in parallel. By running multiple simulations simultaneously, it streamlines and expedites large-scale analyses.
#'
#' @usage
#' run_chain_betas(
#'   N,
#'   gamma_prior=0.3,
#'   iterations,
#'   x_vars,
#'   y,
#'   theta_start=NULL,
#'   size_start=NULL,
#'   use_data_priors=TRUE,
#'   user_fixed_priors=NULL,
#'   dist="ZIP",
#'   epsilon=NULL,
#'   distance_metric="manhattan",
#'   mc_cores = 1
#' )
#'
#' @param y A \code{list} of Hi-C interaction count matrices. Each element of the list
#'   should be an \eqn{N \times N} numeric matrix representing interaction counts between genomic loci.
#'   These data are independently modeled, allowing the function to run separate MCMC chains for each dataset.
#'
#' @param N An integer specifying the dimension of each interaction matrix (\eqn{N \times N}).
#'   All interaction matrices in \code{y} must have this dimension.
#'
#' @param gamma_prior A numeric value specifying the initial (prior) value for the Potts model interaction parameter \eqn{\gamma}.
#'   This parameter influences the spatial dependency structure in the model.
#'
#' @param iterations An integer indicating the number of MCMC iterations to run for each dataset.
#'   During these iterations, parameters (betas, gamma, theta, size) and component (z) are updated repeatedly.
#'
#' @param x_vars A list of covariates (e.g., distance, GC, TES, ACC) used in the regression model for the Hi-C interactions.
#'   Each element of \code{x_vars} is expected to be a list or matrix of appropriate dimensions matching the \eqn{N \times N} lattice.
#'   These covariates are used to adjust for various genomic biases in interaction frequencies.
#'
#' @param theta_start A numeric value for the initial theta parameter (zero-inflation probability) for ZIP or ZINB distributions.
#'   This parameter controls the degree of zero-inflation in the model. If the chosen distribution does not involve zero-inflation,
#'   \code{theta_start} may be ignored.
#'
#'
#' @param size_start A numeric vector of length 3 providing initial values for the size (overdispersion) parameter,
#'   required when \code{dist} is Negative Binomial (NB) or Zero-Inflated Negative Binomial (ZINB).
#'   Each of the three values corresponds to one mixture component.
#'
#' @param use_data_priors data driven prior.
#'
#' @param user_fixed_priors user-specified prior.
#'
#' @param dist A character string specifying the distribution family to use for modeling interaction counts.
#'   Options include:
#'   \itemize{
#'     \item \code{"Poisson"}: Poisson distribution
#'     \item \code{"NB"}: Negative Binomial distribution
#'     \item \code{"ZIP"}: Zero-Inflated Poisson distribution
#'     \item \code{"ZINB"}: Zero-Inflated Negative Binomial distribution
#'   }
#'
#' @param epsilon (Optional) A numeric value for the ABC threshold \eqn{\epsilon}. If \code{NULL}, the threshold is
#'   determined dynamically within the MCMC. This threshold is used to accept or reject proposed \eqn{\gamma} values
#'   based on their ability to produce synthetic data that match observed data according to a given distance metric.
#'
#' @param distance_metric A character string specifying the distance measure used in the ABC step.
#'   Supported values include:
#'   \itemize{
#'     \item \code{"manhattan"}: Absolute differences between means
#'     \item \code{"euclidean"}: Squared differences between means
#'   }
#'
#'
#' @param mc_cores An integer specifying the number of CPU cores to use for parallel processing.
#'   If \code{mc_cores > 1}, the function runs MCMC chains for multiple datasets concurrently,
#'   utilizing multi-core capabilities to reduce total runtime.
#'   Defaults to 1 (no parallelization).
#'
#' @details
#' The \code{run_chain_betas} function provides a convenient interface for running MCMC-based inference
#' on multiple Hi-C datasets simultaneously. It calls \code{run_metropolis_MCMC_betas}
#' which performs:
#' \enumerate{
#'   \item Estimation of mixture model parameters (betas) that adjust for biases like genomic distance, GC content, TES, and accessibility.
#'   \item Updating the Potts model interaction parameter \eqn{\gamma} via Approximate Bayesian Computation (ABC).
#'   \item Handling of zero-inflation (\eqn{\theta}) if using ZIP or ZINB distributions.
#'   \item Updating the dispersion parameter (\eqn{\text{size}}) for NB or ZINB distributions.
#' }
#'
#' By passing a list of count matrices through \code{y}, users can run multiple independent MCMC chains concurrently,
#' facilitating simulation studies, sensitivity analyses, or other large-scale inference tasks.
#'
#' @return
#' A list of length equal to the length of \code{y}. Each element of the returned list is itself a list
#' (as returned by \code{run_metropolis_MCMC_betas}) containing:
#' \itemize{
#'   \item \code{chains}: MCMC samples of the regression parameters for each mixture component.
#'   \item \code{gamma}: Samples of the Potts model interaction parameter.
#'   \item \code{theta}: Samples of the zero-inflation parameter (if applicable).
#'   \item \code{size}: Samples of the dispersion parameter (if using NB or ZINB).
#' }
#'
#' @examples
#' #
#' # Suppose we have multiple simulated Hi-C data sets:
#' N <- 5
#' y <- list(
#'   matrix(rpois((N*N), lambda = 5), nrow = N)
#' )
#'
#'
#' gamma_prior <- 0.3
#' iterations <- 10
#' x_vars <- list(
#'   distance = list(matrix(runif(N * N, 0, 1), nrow = N)),
#'   GC = list(matrix(runif(N * N, 0, 1), nrow = N)),
#'   TES = list(matrix(runif(N * N, 0, 1), nrow = N)),
#'   ACC = list(matrix(runif(N * N, 0, 1), nrow = N))
#' )
#' theta_start <- 0.5
#' size_start <- c(1, 1, 1)
#'
#' # Run the MCMC chains in parallel using 2 cores
#' results <- run_chain_betas(
#'   N = N,
#'   gamma_prior = gamma_prior,
#'   iterations = iterations,
#'   x_vars = x_vars,
#'   y = y,
#'   theta_start = theta_start,
#'   use_data_priors = TRUE,
#'   user_fixed_priors = FALSE,
#'   epsilon = NULL,
#'   distance_metric = "manhattan",
#'   dist = "ZINB",
#'   size_start = size_start,
#'   mc_cores = 1
#' )
#' print(results)
#' # Each element of 'results' corresponds to the output of run_metropolis_MCMC_betas for each dataset
#' #
#'
#' #/donttest{
#' 
#' user_fixed_priors <- list(
#'  component1 = list(
#'    meany = 3, meanx1 = 5, meanx2 = 0.2, meanx3 = 0, meanx4 = 4,
#'    sdy = 1, sdx1 = 0.01, sdx2 = 0.01, sdx3 = 0.002, sdx4 = 0.005
#'  ),
#'  component2 = list(
#'    meany = 300, meanx1 = 8, meanx2 = 0.3, meanx3 = 3, meanx4 = 8,
#'    sdy = 0.1, sdx1 = 0.1, sdx2 = 0.01, sdx3 = 0.02, sdx4 = 0.02
#'  ),
#'  component3 = list(
#'    meany = 200, meanx1 = 10, meanx2 = 0.3, meanx3 = 1.6, meanx4 = 4.5,
#'    sdy = 0.1, sdx1 = 0.6, sdx2 = 0.1, sdx3 = 0.2, sdx4 = 0.2
#'  )
#'  )
#' 
#' #}
#'
#'
#' @seealso [run_metropolis_MCMC_betas()]
#'
#'@importFrom parallel mclapply
#'
#' @export
#'
run_chain_betas <- function(N,
                            gamma_prior = 0.3,
                            iterations,
                            x_vars,
                            y,
                            theta_start = NULL,
                            size_start = NULL,
                            use_data_priors = TRUE,
                            user_fixed_priors = NULL,
                            dist = "ZIP",
                            epsilon = NULL,
                            distance_metric = "manhattan",
                            mc_cores = 1) {
  # y must be a list of matrices
  if (!is.list(y)) stop("'y' must be a list of N by N matrices.")

  # dispatch each element of y to its own chain
  results <- mclapply(seq_along(y), function(i) {
    run_metropolis_MCMC_betas(
      N = N,
      gamma_prior = gamma_prior,
      iterations = iterations,
      x_vars = x_vars,
      y = y[[i]],
      use_data_priors = use_data_priors,
      user_fixed_priors = user_fixed_priors,
      dist = dist,
      epsilon = epsilon,
      distance_metric = distance_metric,
      theta_start = theta_start,
      size_start = size_start
    )
  }, mc.cores = mc_cores)

  return(results)
}
