#' Identify off-targets with bulges for target-specific gRNAs designed for 
#' CRISPR-Cas9 systems.
#' 
#' This function extends the off-targets identified by offTargetAnalysis() by 
#' detecting off-targets that contain bulges. In gRNA design, "bulges" refer to
#' insertions ("RNA bulges") or deletions ("DNA bulges") in the gRNA sequence 
#' relative to the target DNA sequence. Bulges can affect the binding affinity 
#' and specificity of the gRNA to its target. The function wraps around 
#' [`Cas-OFFinder`](http://www.rgenome.net/cas-offinder/) internally.
#'
#' @param gRNA_PAM A `DNAStringSet` object returned by `findgRNA()` that contains
#' gRNA plus PAM sequences. Alternatively, you can supply the `list` object 
#' returned by the `offTargetAnalysis()` function.
#' @param output_csv_name A string specifying the output CSV file name. Defaults
#' to `NULL`, meaning that the output will be printed to the console.
#' @param PAM.size See `offTargetAnalysis()`.
#' @param PAM.pattern See `offTargetAnalysis()`.
#' @param PAM.location See `offTargetAnalysis()`.
#' @param max.mismatch See `offTargetAnalysis()`.
#' @param DNA_bulge The maximum size of DNA bulges, specified in nucleotides. 
#' Defaults to 2.
#' @param RNA_bulge The maximum size of RNA bulges, specified in nucleotides. 
#' Defaults to 2.
#' @param BSgenomeName See `offTargetAnalysis()`. Alternatively, use 
#' `genomeSeqFile` to specify the file path to custom genome fasta file. Note,
#' `genomeSeqFile` overwrites `BSgenomeName` if both set.
#' @param genomeSeqFile If you are using a custom genome, specify the file
#' path to the FASTA file using `genomeSeqFile`.
#' @param chromToSearch See `offTargetAnalysis()`.
#' @param chromToExclude See `offTargetAnalysis()`.
#' @param cas_offinder_version The version of "Cas-OFFinder" to use. 
#' Currently supported versions are "2.4.1" and "3.0.0b3". Defaults to "2.4.1".
#' @return If `output_csv_name` is not set, the function returns a data frame
#' containing the output generated by `Cas-OFFinder`. Otherwise, it saves the
#' data frame to the CSV file specified by `output_csv_name`. When
#' `cas_offinder_version == "2.4.1"`, the following columns will be included: "bulge_type", "gRNA", "DNA", "chr", "start_0_based", "strand", "mismatches", "bulge_size". For `cas_offinder_version == "3.0.0b3"`, the included columns will be: "gRNA_id", "bulge_type", "gRNA", "DNA", "chr", "start_0_based", "strand", "mismatches", "bulge_size".
#' 
#' @seealso
#' `offTargetAnalysis()` for off-targets analysis,
#' `Cas-OFFinder` (https://github.com/snugel/cas-offinder) for more on output format.
#' @references 
#' 1. Sangsu Bae, Jeongbin Park, Jin-Soo Kim, Cas-OFFinder: a fast and versatile algorithm that searches for potential off-target sites of Cas9 RNA-guided endonucleases, Bioinformatics, Volume 30, Issue 10, May 2014, Pages 1473–1475, https://doi.org/10.1093/bioinformatics/btu048
#' @author Kai Hu
#' 
#' @examples
#' # Example with `DNAStringSet` as input
#' if (interactive()) {
#'   library(CRISPRseek)
#'   library(BSgenome.Hsapiens.UCSC.hg19)
#' 
#'   gRNA_PAM <- findgRNAs(inputFilePath = system.file("extdata", 
#'                                                     "inputseq.fa", 
#'                                                      package = "CRISPRseek"),
#'                         pairOutputFile = "testpairedgRNAs.xls",
#'                         findPairedgRNAOnly = TRUE)
#'   df <- getOfftargetWithBulge(gRNA_PAM, PAM.pattern = "NNG$|NGN$",
#'                              DNA_bulge = 2, RNA_bulge = 2,
#'                              BSgenomeName = Hsapiens, chromToSearch = "chrX")
#' 
#'  # Example with `list` output from `offTargetAnalysis` as input
#'  library(TxDb.Hsapiens.UCSC.hg19.knownGene)
#'  library(org.Hs.eg.db)
#' 
#'  inputFilePath <- system.file("extdata", "inputseq.fa", package = "CRISPRseek")
#'  REpatternFile <- system.file("extdata", "NEBenzymes.fa", package = "CRISPRseek")
#'  res <- offTargetAnalysis(inputFilePath, 
#'                           findgRNAsWithREcutOnly = TRUE,
#'                           REpatternFile = REpatternFile, 
#'                           findPairedgRNAOnly = FALSE, 
#'                           annotatePaired = FALSE,
#'                           BSgenomeName = Hsapiens, 
#'                           chromToSearch = "chrX",
#'                           txdb = TxDb.Hsapiens.UCSC.hg19.knownGene,
#'                           orgAnn = org.Hs.egSYMBOL, max.mismatch = 1,
#'                           outputDir = tempdir(),
#'                           overwrite = TRUE)
#'  df <- getOfftargetWithBulge(res, PAM.pattern = "NNG$|NGN$",
#'                              DNA_bulge = 2, 
#'                              RNA_bulge = 2, 
#'                              BSgenomeName = Hsapiens, 
#'                              chromToSearch = "chrX") 
#'  }
#' @importFrom rlang abort inform
#' @export
getOfftargetWithBulge <- function(gRNA_PAM = NULL,
                                  output_csv_name = NULL,
                                  PAM.size = 3,
                                  PAM.pattern = "NNG$|NGN$",
                                  PAM.location = c("3prime", "5prime"),
                                  max.mismatch = 3,
                                  DNA_bulge = 2,
                                  RNA_bulge = 2,
                                  BSgenomeName = NULL,
                                  genomeSeqFile = NULL,
                                  chromToSearch = "all",
                                  chromToExclude = NULL,
                                  cas_offinder_version = c("2.4.1", "3.0.0b3")) {
  # Validate parameters:
  PAM.location <- match.arg(PAM.location)
  cas_offinder_version <- match.arg(cas_offinder_version)
  if (inherits(gRNA_PAM, "DNAStringSet")) {
    # inform("gRNA_PAM: DNAStringSet supplied")
    if (is.null(names(gRNA_PAM))) {
      names(gRNA_PAM) <- seq(1:length(gRNA_PAM))
    }
  } else if (inherits(gRNA_PAM, "list")) {
    # inform("gRNA_PAM: offTargetAnalysis results supplied")
    if (!("summary" %in% names(gRNA_PAM))) {
      stop("Not an offTargetAnalysis() result object!")
    }
    gRNA_PAM <- getgRNAFromoffTargetAnalysis(gRNA_PAM)
  }
  if (is.null(BSgenomeName) && is.null(genomeSeqFile)) {
    stop("Either 'BSgenomeName' or 'genomeSeqFile' must be provided.")
  } else if (!is.null(genomeSeqFile) && !file.exists(genomeSeqFile)) {
    stop("The specified 'genomeSeqFile' does not exist.")
  } else if (!is.null(BSgenomeName) && !inherits(BSgenomeName, "BSgenome")) {
    stop("'BSgenomeName' must be a valid BSgenome object.")
  }
  
  # Check CasOFFinder installation, returns device type: "G" or "C"
  binary_path <- getCasOFFinder(version = cas_offinder_version)
  device <- checkCasOFFinder(binary_path = binary_path)
  
  # Prepare CasOFFinder wrapper function:
  cas_offinder <- prepCasOFFinderBulge(version = cas_offinder_version)
  
  # Prepare CasOFFinder input genome files:
  chr_path <- prepCasOFFinderInputGenome(BSgenomeName = BSgenomeName,
                                         genomeSeqFile = genomeSeqFile,
                                         chromToSearch = chromToSearch, 
                                         chromToExclude = chromToExclude,
                                         outdir = tempdir())
  
  # Prepare CasOFFinder input.txt:
  input_txts <- prepCasOFFinderInputTxt(gRNA_PAM = gRNA_PAM,
                                        PAM.size = PAM.size,
                                        PAM.location = PAM.location,
                                        max.mismatch = max.mismatch,
                                        DNA_bulge = DNA_bulge,
                                        RNA_bulge = RNA_bulge,
                                        chr_path = chr_path,
                                        cas_offinder_version = cas_offinder_version,
                                        outdir = tempdir())
  
  # Run CasOFFinder:
  df <- data.frame()
  for (i in seq_along(input_txts)) {
    output_name <- file.path(tempdir(), paste0("output", i, "_", basename(tempfile(pattern = "")), ".txt"))
    cas_offinder(input_txts[[i]], device, output_name, binary_path)
    df_tem <- read.table(output_name, sep = "\t", header = FALSE)
    if (cas_offinder_version == "2.4.1") {
      colnames(df_tem) <- c("bulge_type", "gRNA", "DNA", "chr", "start_0_based", "strand", "mismatches", "bulge_size")
    } else if (cas_offinder_version == "3.0.0b3") {
      colnames(df_tem) <- c("gRNA_id", "bulge_type", "gRNA", "DNA", "chr", "start_0_based", "strand", "mismatches", "bulge_size")
    }
    df <- rbind(df_tem, df)
  }
  
  if (is.null(output_csv_name)) {
    return(df)
    # inform("Done!")
  } else {
    write.table(df, file = output_csv_name, sep = ",", row.names = FALSE)
    # inform(paste0("Done! Results saved at ", output_csv_name, "."))
  }
}