#' @title Validate input parameters for createAccuracyGraph 
#' function
#'
#' @description This function validates the parameters for the 
#' \code{\link{createAccuracyGraph}} function.
#' 
#' @param dfAUROC a \code{data.frame} corresponding to res$paraSample$dfAUROC
#' where res is the result of inferAncestry() or inferAncestryGeneAware() 
#' functions.
#'
#' @param title a \code{character} string representing the title of the graph.
#' 
#' @param selectD a \code{array} of \code{integer} representing the selected 
#' PCA dimensions to plot. The length of the \code{array} cannot be more than 
#' 5 entries. The dimensions must tested by RAIDS (i.e. be present in the 
#' RDS file). 
#' 
#' @param selectColor a \code{array} of \code{character} strings representing 
#' the selected colors for the associated PCA dimensions to plot. The length 
#' of the \code{array} must correspond to the length of the \code{selectD} 
#' parameter. In addition, the length of the \code{array} cannot be more than 
#' 5 entries. 
#'  
#' @return The function returns \code{0L} when successful.
#'
#' @examples
#' 
#' ## Path to RDS file with ancestry information generated by RAIDS (demo file)
#' dataDir <- system.file("extdata", package="RAIDS")
#' fileRDS <- file.path(dataDir, "TEST_01.infoCall.RDS")
#' info <- readRDS(fileRDS)
#' dfAUROC <- info$paraSample$dfAUROC
#' 
#' ## Some of the column names must be updated to fit new standards
#' colnames(dfAUROC) <- c("D", "K", "Call", "L", "AUROC", "H")
#' 
#' ## Validate parameters
#' RAIDS:::validatecreateAUROCGraph(dfAUROC=dfAUROC, title="Accuracy Graph", 
#'     selectD=c(6, 12), selectColor=c("blue","darkblue"))
#'     
#' @author Astrid Deschênes and Pascal Belleau
#' @importFrom stringr str_detect
#' @encoding UTF-8
#' @keywords internal
validatecreateAUROCGraph <- function(dfAUROC, title, selectD, selectColor) {
    
    ## Validate dfAUROC is a data frame
    if(!is.data.frame(dfAUROC)) {
        stop("The \'dfAUROC\' parameter must be a data frame.")
    }
    
    ## Validate the file extension
    colL <- c("D", "K", "Call", "L", "AUROC", "H")
    if (!all(colL %in% colnames(dfAUROC))) {
        stop("The \'dfAUROC\' must have all those columns: D, K, Call, ", 
                "L, AUROC, H.")
    }
    
    ## Validate other parameters
    res <- validateAccuracyGraphInternal(title=title, selectD=selectD, 
                                                selectColor=selectColor)
    
    ## Success
    return(res)
}


#' @title Validate input parameters for createAccuracyGraph 
#' function
#'
#' @description This function validates the parameters for the 
#' \code{\link{createAccuracyGraph}} function.
#' 
#' @param fileRDS a \code{character} string representing the path and file
#' name of the RDS file containing the ancestry information as generated by 
#' RAIDS.
#'
#' @param title a \code{character} string representing the title of the graph.
#' 
#' @param selectD a \code{array} of \code{integer} representing the selected 
#' PCA dimensions to plot. The length of the \code{array} cannot be more than 
#' 5 entries. The dimensions must tested by RAIDS (i.e. be present in the 
#' RDS file). 
#' 
#' @param selectColor a \code{array} of \code{character} strings representing 
#' the selected colors for the associated PCA dimensions to plot. The length 
#' of the \code{array} must correspond to the length of the \code{selectD} 
#' parameter. In addition, the length of the \code{array} cannot be more than 
#' 5 entries. 
#'  
#' @return The function returns \code{0L} when successful.
#'
#' @examples
#' 
#' ## Path to RDS file with ancestry information generated by RAIDS (demo file)
#' dataDir <- system.file("extdata", package="RAIDS")
#' fileRDS <- file.path(dataDir, "TEST_01.infoCall.RDS")
#' 
#' ## Validate parameters
#' RAIDS:::validateCreateAccuracyGraph(fileRDS=fileRDS, title="Accuracy Graph", 
#'     selectD=c(5, 10), selectColor=c("blue","darkblue"))
#'     
#' @author Astrid Deschênes and Pascal Belleau
#' @importFrom stringr str_detect
#' @encoding UTF-8
#' @keywords internal
validateCreateAccuracyGraph <- function(fileRDS, title, selectD, selectColor) {
    
    ## Validate fileRDS is character string
    if(!is.character(fileRDS)) {
        stop("The \'fileRDS\' parameter must be a character string.")
    }
    
    ## Validate fileRDS exist
    if(!file.exists(fileRDS)) {
        stop("The \'fileRDS\' file does not exist.")
    }
    
    ## Validate the file extension
    if (!str_detect(fileRDS, "\\.(RDS|rds)$")) {
        stop("The \'fileRDS\' must have a RDS (or rds) extension.")
    }
    
    ## Validate other parameters
    res <- validateAccuracyGraphInternal(title=title, selectD=selectD, 
                                         selectColor=selectColor)
    
    ## Success
    return(res)
}


#' @title Validate input parameters for createAccuracyGraph and 
#' createAUROCGraph functions
#'
#' @description This function validates the parameters for the 
#' \code{\link{createAccuracyGraph}} and \code{\link{createAUROCGraph}} 
#' functions.
#' 
#' @param title a \code{character} string representing the title of the graph.
#' 
#' @param selectD a \code{array} of \code{integer} representing the selected 
#' PCA dimensions to plot. The length of the \code{array} cannot be more than 
#' 5 entries. The dimensions must tested by RAIDS (i.e. be present in the 
#' RDS file). 
#' 
#' @param selectColor a \code{array} of \code{character} strings representing 
#' the selected colors for the associated PCA dimensions to plot. The length 
#' of the \code{array} must correspond to the length of the \code{selectD} 
#' parameter. In addition, the length of the \code{array} cannot be more than 
#' 5 entries. 
#'  
#' @return The function returns \code{0L} when successful.
#'
#' @examples
#' 
#' ## Validate parameters
#' RAIDS:::validateAccuracyGraphInternal(title="Accuracy Graph", 
#'     selectD=c(5, 10), selectColor=c("blue","darkblue"))
#'     
#' @author Astrid Deschênes and Pascal Belleau
#' @importFrom stringr str_detect
#' @encoding UTF-8
#' @keywords internal
validateAccuracyGraphInternal <- function(title, selectD, selectColor) {
    
    ## Validate the title is a string
    if (!is.character(title)) {
        stop("The \'title\' must be a character string.")
    }
    
    ## Validate the length of the selectD parameter
    if (length(selectD) == 0) {
        stop("The \'selectD\' parameter cannot be empty.")
    }
    
    ## Validate the length of the selectD parameter
    if (length(selectD) > 5) {
        stop("The \'selectD\' parameter can contain a maximum of 5 elements.")
    }
    
    ## Validate lengths of selectD and selectcolor identical
    if (length(selectColor) != length(selectD)) {
        stop("The \'selectColor\' parameter must be the same length than ", 
                "the \'selectD\' parameter.")
    }
    
    ## Success
    return(0L)
}