% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/differential-expression.R
\name{addDifferentialExpression}
\alias{addDifferentialExpression}
\title{Manually add differential expression results to a MirnaExperiment object}
\usage{
addDifferentialExpression(
  mirnaObj,
  mirnaDE = NULL,
  geneDE = NULL,
  mirna.logFC = 0,
  mirna.pCutoff = 0.05,
  mirna.pAdjustment = "fdr",
  gene.logFC = 0,
  gene.pCutoff = 0.05,
  gene.pAdjustment = "fdr"
)
}
\arguments{
\item{mirnaObj}{A \code{\link[=MirnaExperiment-class]{MirnaExperiment}} object
containing miRNA and gene data}

\item{mirnaDE}{A \code{data.frame} containing the output of miRNA differential
expression analysis. Check the \emph{details} section to see the required format.
Default is NULL not to add miRNA differential expression results}

\item{geneDE}{A \code{data.frame} containing the output of gene differential
expression analysis. Check the \emph{details} section to see the required format.
Default is NULL not to add gene differential expression results}

\item{mirna.logFC}{The minimum log2 fold change required to consider a miRNA
as differentially expressed. Optional, default is 0}

\item{mirna.pCutoff}{The adjusted p-value cutoff to use for miRNA statistical
significance. The default value is \code{0.05}}

\item{mirna.pAdjustment}{The p-value correction method for miRNA multiple
testing. It must be one of: \code{fdr} (default), \code{BH}, \code{none}, \code{holm},
\code{hochberg}, \code{hommel}, \code{bonferroni}, \code{BY}}

\item{gene.logFC}{The minimum log2 fold change required to consider a gene as
differentially expressed. Optional, default is 0}

\item{gene.pCutoff}{The adjusted p-value cutoff to use for gene statistical
significance. The default value is \code{0.05}}

\item{gene.pAdjustment}{The p-value correction method for gene multiple
testing. It must be one of: \code{fdr} (default), \code{BH}, \code{none}, \code{holm},
\code{hochberg}, \code{hommel}, \code{bonferroni}, \code{BY}}
}
\value{
A \code{\link[=MirnaExperiment-class]{MirnaExperiment}} object containing differential
expression results. To access these results, the user may run the
\code{\link[=mirnaDE]{mirnaDE()}} and \code{\link[=geneDE]{geneDE()}} functions for miRNAs and genes, respectively.
}
\description{
This function allows to add miRNA and gene differential expression results
to a \code{\link[=MirnaExperiment-class]{MirnaExperiment}} object. Instead of running
\code{\link[=performMirnaDE]{performMirnaDE()}} and \code{\link[=performGeneDE]{performGeneDE()}} functions, this one allows to use
differential expression analyses carried out in other ways. Note that it is
possible to manually add differential expression results just for miRNAs or
just for genes. This is particularly useful in order to use the pipeline
implemented in MIRit for proteomic data and for expression data deriving
from different technologies.
}
\details{
The following paragraphs briefly explain the formats needed for mirnaDE,
geneDE, and differential expression parameters.
\subsection{mirnaDE and geneDE}{

\code{mirnaDE} and \code{geneDE} are two objects of class \code{data.frame} containing
the results of miRNA and gene differential expression analysis respectively.
These tables should contain the differential expression results for all
miRNAs/genes analyzed, not just for statistically significant species.

Note that you can individually add differential expression results for
miRNAs and genes. For instance, it is possible to manually add gene
differential expression through this function, while performing miRNA
differential expression through the \code{\link[=performMirnaDE]{performMirnaDE()}} function, and vice
versa. In order to only add miRNA or gene differential expression results,
you must leave \code{mirnaDE} or \code{geneDE} slots to NULL.

All \code{data.frame} objects can be used, as long as they have:
\itemize{
\item One column containing miRNA/gene names (according to miRBase/hgnc
nomenclature). Accepted column names are: \code{ID}, \code{Symbol}, \code{Gene_Symbol},
\code{Mirna}, \code{mir}, \code{Gene}, \code{gene.symbol}, \code{Gene.symbol};
\item One column with log2 fold changes. Accepted column names are: \code{logFC},
\code{log2FoldChange}, \code{FC}, \code{lFC};
\item One column with average expression. Accepted column names are: \code{AveExpr},
\code{baseMean}, \code{logCPM};
\item One column with the p-values resulting from the differential expression
analysis. Accepted column names are: \code{P.Value}, \code{pvalue}, \code{PValue},
\code{Pvalue};
\item One column containing p-values adjusted for multiple testing. Accepted
column names are: \code{adj.P.Val}, \code{padj}, \code{FDR}, \code{fdr}, \code{adj}, \code{adj.p}, \code{adjp}.
}
}

\subsection{Differential expression cutoffs}{

\code{mirna.logFC}, \code{mirna.pCutoff}, \code{mirna.pAdjustment}, and \code{gene.logFC},
\code{gene.pCutoff}, \code{gene.pAdjustment} represent the parameters used to define
the significance of differential expression results. These are needed in
order to inform MIRit about the features that are considered as
differentially expressed.
}
}
\examples{
# load example data
data(geneCounts, package = "MIRit")
data(mirnaCounts, package = "MIRit")

# create samples metadata
meta <- data.frame(
    "primary" = colnames(geneCounts),
    "mirnaCol" = colnames(mirnaCounts), "geneCol" = colnames(geneCounts),
    "disease" = c(rep("PTC", 8), rep("NTH", 8)),
    "patient" = c(rep(paste("Sample_", seq(8), sep = ""), 2))
)

# create a 'MirnaExperiment' object
obj <- MirnaExperiment(
    mirnaExpr = mirnaCounts, geneExpr = geneCounts,
    samplesMetadata = meta, pairedSamples = TRUE
)

# load example tables with differential expression results
de_m <- mirnaDE(object = loadExamples(), onlySignificant = FALSE)
de_g <- geneDE(object = loadExamples(), onlySignificant = FALSE)

# add DE results to MirnaExperiment object
obj <- addDifferentialExpression(obj, de_m, de_g,
    mirna.pCutoff = 0.05,
    gene.pCutoff = 0.05
)

}
\author{
Jacopo Ronchi, \email{jacopo.ronchi@unimib.it}
}
