## ----style, echo = FALSE, results = 'asis', message=FALSE---------------------
BiocStyle::markdown()

## ----echo = FALSE, message = FALSE--------------------------------------------
library(Chromatograms)
library(BiocStyle)
register(SerialParam())

## -----------------------------------------------------------------------------
# A data.frame with chromatogram variables.
cdata <- data.frame(
    msLevel = c(1L, 1L),
    mz = c(112.2, 123.3),
    chromIndex = c(1L, 2L)
)

# Retention time and intensity values for each chromatogram.
pdata <- list(
    data.frame(
        rtime = c(11, 12.4, 12.8, 13.2, 14.6, 15.1, 16.5),
        intensity = c(50.5, 123.3, 153.6, 2354.3, 243.4, 123.4, 83.2)
    ),
    data.frame(
        rtime = c(45.1, 46.2, 53, 54.2, 55.3, 56.4, 57.5),
        intensity = c(100, 180.1, 300.45, 1400, 1200.3, 300.2, 150.1)
    )
)

# Create and initialize the backend
be <- backendInitialize(ChromBackendMemory(),
    chromData = cdata, peaksData = pdata
)

# Create Chromatograms object
chr <- Chromatograms(be)
chr

## -----------------------------------------------------------------------------
MRM_file <- system.file("proteomics", "MRM-standmix-5.mzML.gz",
    package = "msdata"
)

be <- backendInitialize(ChromBackendMzR(),
    files = MRM_file,
    BPPARAM = SerialParam()
)

chr_mzr <- Chromatograms(be)

## -----------------------------------------------------------------------------
length(chr)
length(chr_mzr)

## -----------------------------------------------------------------------------
peaksData(chr)

## -----------------------------------------------------------------------------
peaksData(chr, columns = c("rtime"), drop = TRUE)

chr$rtime

## -----------------------------------------------------------------------------
peaksData(chr) <- list(
    data.frame(
        rtime = c(1, 2, 3, 4, 5, 6, 7),
        intensity = c(1, 2, 3, 4, 5, 6, 7)
    ),
    data.frame(
        rtime = c(1, 2, 3, 4, 5, 6, 7),
        intensity = c(1, 2, 3, 4, 5, 6, 7)
    )
)

## -----------------------------------------------------------------------------
chr$rtime <- list(
    c(8, 9, 10, 11, 12, 13, 14),
    c(8, 9, 10, 11, 12, 13, 14)
)

## -----------------------------------------------------------------------------
chr_filt <- filterPeaksData(chr, variables = "rtime", ranges = c(12, 15))

length(chr_filt)

length(rtime(chr_filt))

## -----------------------------------------------------------------------------
chromData(chr)

## -----------------------------------------------------------------------------
chromData(chr, columns = c("msLevel"))

chr$chromIndex

## -----------------------------------------------------------------------------
chr$msLevel <- c(2L, 2L)

chromData(chr)

## -----------------------------------------------------------------------------
chr$extra <- c("extra1", "extra2")
chromData(chr)

## -----------------------------------------------------------------------------
chr_filt <- filterChromData(chr,
    variables = "chromIndex", ranges = c(1, 2),
    keep = TRUE
)

length(chr_filt)
length(chr)

## ----define-function----------------------------------------------------------
## Define a function that takes the backend as an input, divides the intensity
## by parameter y and returns it. Note that ... is required in
## the function's definition.
divide_intensities <- function(x, y, ...) {
    intensity(x) <- lapply(intensity(x), `/`, y)
    x
}

## Add the function to the procesing queue
chr_2 <- addProcessing(chr, divide_intensities, y = 2)
chr_2

## ----custom-processing--------------------------------------------------------
intensity(chr_2)
intensity(chr)

## ----applyProcessing----------------------------------------------------------
length(chr_2@processingQueue)
chr_2 <- applyProcessing(chr_2)

length(chr_2@processingQueue)
chr_2

## -----------------------------------------------------------------------------
processingChunkFactor(chr)

## -----------------------------------------------------------------------------
processingChunkFactor(chr_mzr) |>
  head()

## -----------------------------------------------------------------------------
processingChunkSize(chr_mzr) <- 10

processingChunkFactor(chr_mzr) |> table()

## -----------------------------------------------------------------------------
print(object.size(chr_mzr), units = "Mb")
chr_mzr <- setBackend(chr_mzr, ChromBackendMemory(), BPPARAM = SerialParam())

chr_mzr

chr_mzr@backend@peaksData[[1]] |> head() # data is now in memory

## -----------------------------------------------------------------------------
print(object.size(chr_mzr), units = "Mb")

## ----message=FALSE------------------------------------------------------------
library(Spectra)
library(MsBackendMetaboLights)
be <- backendInitialize(MsBackendMetaboLights(),
    mtblsId = "MTBLS39",
    filePattern = c("63B.cdf")
)
chr_s <- Chromatograms(Spectra(be))

## -----------------------------------------------------------------------------
chr_s

## -----------------------------------------------------------------------------
chromData(chr_s)$rtmin <- 125
chromData(chr_s)$rtmax <- 180
chromData(chr_s)$mzmin <- 100
chromData(chr_s)$mzmax <- 100.5

## -----------------------------------------------------------------------------
library(RColorBrewer)
col3 <- brewer.pal(3, "Dark2")

plotChromatograms(chr_s, col = col3)

## -----------------------------------------------------------------------------
plotChromatogramsOverlay(chr_s, col = col3)

## ----si-----------------------------------------------------------------------
sessionInfo()

