## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(
    cache = TRUE,
    comment = "#>"
)
library(BiocStyle)

## ----install, eval=FALSE------------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# 
# BiocManager::install("Bioc.gff")

## ----load---------------------------------------------------------------------
library(Bioc.gff)
library(GenomicRanges)

## ----find-gff-----------------------------------------------------------------
gff_file <- system.file("extdata", "genes.gff3", package = "Bioc.gff")

## ----read-gff-----------------------------------------------------------------
import(gff_file)

## ----read-gff-which-----------------------------------------------------------
which <- GRanges("chr10:90000-93000")
import(gff_file, which = which, genome = "hg19")

## ----read-gff-version---------------------------------------------------------
readGFF(gff_file, version = "3")

## ----read-gff-granges---------------------------------------------------------
readGFF(gff_file, version = "3") |>
    makeGRangesFromDataFrame(
        keep.extra.columns = TRUE
    )

## ----read-gff-remote-direct,eval=FALSE----------------------------------------
# remote_gff_url <- "https://www.mirbase.org/download/hsa.gff3"
# import(remote_gff_url, version = "3")

## ----read-gff-remote----------------------------------------------------------
library(BiocFileCache)
bfc <- BiocFileCache::BiocFileCache()
remote_gff_url <- "https://www.mirbase.org/download/hsa.gff3"
bquery <- bfcquery(bfc, remote_gff_url, "rname", exact = TRUE)
if (!nrow(bquery))
    bfcadd(x = bfc, rname = remote_gff_url, rtype = "web", download = TRUE)
gff_local <- bfcrpath(
    bfc, rnames = remote_gff_url, exact = TRUE, download = FALSE, rtype = "web"
)

## ----read-gff-remote2---------------------------------------------------------
import(gff_local, version = "3")

## ----as-gff-------------------------------------------------------------------
library(GenomicFeatures)
library(TxDb.Hsapiens.UCSC.hg19.knownGene)
txdb <- TxDb.Hsapiens.UCSC.hg19.knownGene
exonsBy(txdb, by = "tx") |>
    asGFF()

## ----make-txdb-from-gff-------------------------------------------------------
library(txdbmaker)
txdb <- makeTxDbFromGFF(
    file = gff_local,
    format = "gff3",
    dataSource = "https://www.mirbase.org/download/hsa.gff3",
    organism = "Homo sapiens",
    taxonomyId = 9606
)
genome <- grepv("genome-build-id", readLines(gff_local)) |>
    strsplit("# genome-build-id:\\s+") |>
    unlist() |>
    tail(1L)
genome(txdb) <- genome
txdb

## ----session-info-------------------------------------------------------------
sessionInfo()

