#' Single-cell transcriptome data of medullary thymic epithelial cells
#' 
#' This document contains all the code used to analyse the single-cell
#' RNA-seq and the bulk ATAC-seq data from the manuscript by Brennecke et al, 2015.
#' The purpose of this package is to provide full reproducibility of the results
#' presented in the manuscript. This package provides a documented and reproducible
#' workflow of the code that was used to generate each number and figure from
#' the manuscript.
#' 
#' @docType package
#' @name Single.mTec.Transcriptomes-package
#' @references
#'
#'  Brennecke et al. Single-cell transcriptome analysis reveals coordinated ectopic gene-expression patterns in medullary thymic epithelial cells. Nature Immunology 16,933-941 (2015)
#' 
NULL

#' List of Aire-dependent genes
#' 
#' A character vector of ensembl gene identifiers defined by Sansom et al, 2014. This list was downloaded from the supplementary material of the manuscript.
#' 
#' @docType data
#' @keywords datasets
#' @name aireDependentSansom
#' @usage data(aireDependentSansom)
#'
#' @references
#'
#' Sansom et al. Population and single-cell genomics reveal the Aire dependency, relief from Polycomb silencing, and distribution of self-antigen expression in thymic epithelia. Genome Res. 24, 1918--1931 (2014). 
#' 
NULL

#' Human biotypes
#' 
#' A character vector defining biotype for each human ensembl gene identifier. Biotypes were queried using \pkg{biomaRt}.
#' 
#' @docType data
#' @keywords datasets
#' @name biotypesHuman
#' @usage data(biotypesHuman)
NULL

#' Mouse biotypes
#' 
#' A character vector defining biotype for each mouse ensembl gene identifier. Biotypes were queried using \pkg{biomaRt}.
#' 
#' @docType data
#' @keywords datasets
#' @name biotype
#' @usage data(biotypes)
NULL

#' Cea1 co-expression group data
#' 
#' A data frame containing information about the Cea1 co-expression group defined in Pinto et al, 2013.
#' 
#' @docType data
#' @keywords datasets
#' @name cea1Coexpression
#' @usage data(cea1Coexpression)
#'
#' @references
#'
#' Pinto et al. Overlapping gene coexpression patterns in human medullary thymic epithelial cells generate self-antigen diversity. Proc. Natl. Acad. Sci. U.S.A. 110, E3497--3505, (2013).
#' 
NULL

#' Gene-gene correlation network
#' 
#' A matrix containing the gene-gene Spearman correlation across single unselected cells.
#' 
#' @docType data
#' @keywords datasets
#' @name corMatsNoMarker
#' @aliases corMatSpNoMarker
#' @usage data(corMatsNoMarker)
#'
NULL

#' Sansom et al's gene-gene correlation network
#' 
#' A matrix containing the gene-gene Spearman correlation across single unselected cells, using the data from Sansom et al, 2014.
#' 
#' @docType data
#' @keywords datasets
#' @name corMatsNoMarker
#' @aliases corMatSp
#' @usage data(corMatsSansom)
#'
#' @references
#' Sansom et al. Population and single-cell genomics reveal the Aire dependency, relief from Polycomb silencing, and distribution of self-antigen expression in thymic epithelia. Genome Res. 24, 1918--1931 (2014). 
#'
NULL

#' Highly variable genes.
#' 
#' List of highly variable genes according to the method by Brennecke et al, 2013.
#' 
#' @docType data
#' @keywords datasets
#' @name deGenesNone
#' @usage data(deGenesNone)
#'
#' @references
#' 
#' Brennecke et al. Accounting for technical noise in single-cell RNA-seq experiments. Nat. Methods. 10, 1093-1095 (2013).
#'
NULL


#' Highly variable genes from Sansom et al.
#' 
#' List of highly variable genes according to the method by Brennecke et al, 2013, using the data by Santom et al.
#' 
#' @docType data
#' @keywords datasets
#' @name deGenesSansom
#' @usage data(deGenesSansom)
#'
#' @references
#' 
#' Brennecke et al. Accounting for technical noise in single-cell RNA-seq experiments. Nat. Methods. 10, 1093-1095 (2013).
#' Sansom et al. Population and single-cell genomics reveal the Aire dependency, relief from Polycomb silencing, and distribution of self-antigen expression in thymic epithelia. Genome Res. 24, 1918--1931 (2014). 
#'
NULL

#' ATAC-seq summarized counts.
#' 
#' DESeqDataSet object summarizing the ATAC-seq data presented in the manuscript. This object contains the read counts of each sample over a window of 4Kb around transcription start sites.
#' 
#' @docType data
#' @keywords datasets
#' @name dxdATAC
#' @usage data(dxdATAC)
#'
NULL

#' FANTOM dataset
#' 
#' DESeqDataSet object of the read counts from selected tissues from the FANTOM dataset.
#' 
#' @docType data
#' @keywords datasets
#' @name fantom
#' @aliases dxdFANTOM
#' @usage data(fantom)
#'
#' @references
#' 
#' Forrest et al. A promoter-level mammalian expression atlas. 24, 1918--1931. (2014).
#'
NULL

#' Mouse gene names
#' 
#' Character vector of mouse gene names. The gene names were queried using \pkg{biomaRt}.
#' 
#' @docType data
#' @keywords datasets
#' @name geneNames
#' @usage data(geneNames)
#'
NULL

#' Human gene names
#' 
#' Character vector of human gene names. The gene names were queried using \pkg{biomaRt}.
#' 
#' @docType data
#' @keywords datasets
#' @name geneNamesHuman
#' @usage data(geneNamesHuman)
#'
NULL

#' Genomic range coordinates.
#' 
#' GenomicRanges object containing the coordinate ranges of mouse protein coding genes used in the manuscript.
#' 
#' @docType data
#' @keywords datasets
#' @name geneRanges
#' @usage data(geneRanges)
#'
NULL

#' Count data from the single-mTEC data.
#' 
#' DESeqDataSet object containing the read counts from the single-cell RNA-seq dataset generated by Brennecke et al, 2015.
#' 
#' @docType data
#' @keywords datasets
#' @name mTECdxd
#' @aliases dxd
#' @usage data(mTECdxd)
#'
#' @references
#' 
#'  Brennecke et al. Single-cell transcriptome analysis reveals coordinated ectopic gene-expression patterns in medullary thymic epithelial cells. Nature Immunology 16,933-941 (2015)
#'
NULL

#' Muc1 co-expression group data
#' 
#' A data frame containing information about the Muc1 co-expression group defined in Pinto et al, 2013.
#' 
#' @docType data
#' @keywords datasets
#' @name muc1Coexpression
#' @usage data(muc1Coexpression)
#'
#' @references
#'
#' Pinto et al. Overlapping gene coexpression patterns in human medullary thymic epithelial cells generate self-antigen diversity. Proc. Natl. Acad. Sci. U.S.A. 110, E3497--3505, (2013).
#' 
NULL

#' Clustering results
#' 
#' List containing the results from the clustering analysis from Figure 2 from the manuscript.
#' 
#' @docType data
#' @keywords datasets
#' @name nomarkerCellsClustering
#' @usage data(nomarkerCellsClustering)
#'
NULL

#' Mapping statistics
#' 
#' Data frame containing the mapping statistics from the single-cell RNA-seq data.
#' 
#' @docType data
#' @keywords datasets
#' @name percentsGG
#' @usage data(percentsGG)
#'
NULL

#' Permutation results
#' 
#' Result from the gene permutations from Figure 5 from the manuscript.
#' 
#' @docType data
#' @keywords datasets
#' @name permutationResults
#' @aliases permsAllClusters realAllClusters
#' @usage data(permutationResults)
#'
NULL

#' Output from scLVM
#' 
#' Objects containing the output scLVM on the single-mTEC RNA-seq data.
#' 
#' @docType data
#' @keywords datasets
#' @name scLVM_output
#' @aliases Ycorr beta beta0 h5GeneNames heter vars
#' @usage data(scLVM_output)
#'
#' @references
#'
#' Buettner et al. Computational analysis of cell-to-cell heterogeneity in single-cell RNA-sequencing data reveals hidden subpopulations of cells. Nat. Biotechnol. 2015.
#'
NULL


#' Tissue restricted antigens lists. 
#' 
#' Data frame containing the TRA classification from Pinto et al, 2013.
#' 
#' @docType data
#' @keywords datasets
#' @name tras
#' @usage data(tras)
#'
NULL
