
#'Summarize the metadata of a SummarizedExperiment object from the DoReMiTra collection
#'
#'#' Provides a concise overview of a `SummarizedExperiment` object from the DoReMiTra collection.
#' Displays information such as the number of samples, metadata content, and
#' key experimental conditions like radiation type, organism, and platform.
#'
#' @param se The name of the SE object as defined by the user
#'
#' @returns A character string containing a essential information about the dataset metadata
#'
#' @export
#'
#' @importFrom S4Vectors metadata
#' @importFrom glue glue

#' @examples
#' se <- get_DoReMiTra_data("SE_Ghandhi_2018_InVivo_GSE84898_GPL13497")
#' summarize_DoReMiTra_se(se)

summarize_DoReMiTra_se <- function(se) {
  # Check that se is a SummarizedExperiment
  if (!inherits(se, "SummarizedExperiment")) {
    stop("Input must be a SummarizedExperiment object.", call. = FALSE)
  }

  # Extract DoReMiTra metadata
  meta <- metadata(se)[["DoReMiTra"]]

  # Check if metadata exists and is a list
  if (is.null(meta) || !is.list(meta)) {
    stop("The SummarizedExperiment metadata is not in the expected 'DoReMiTra' format.", call. = FALSE)
  }

  # Define expected fields
  expected_fields <- c("Author", "Organism", "Radiation", "Setting", "Accession", "Link")

  # Check for missing fields
  missing_fields <- setdiff(expected_fields, names(meta))
  if (length(missing_fields) > 0) {
    stop(
      paste0(
        "The 'DoReMiTra' metadata is missing the following fields: ",
        paste(missing_fields, collapse = ", ")
      ),
      call. = FALSE
    )
  }

  author <- meta$Author
  n_samples <- ncol(se)
  platform <- unique(se$Platform)
  species <- meta$Organism
  radiation <- meta$Radiation
  setting <- meta$Setting
  accession <- meta$Accession
  link <- meta$Link

  message(glue::glue(
    "\nThis dataset is generated by: {author}
    Platform: {platform}
    Organism(s): {species}
    Radiation Type: {radiation}
    Experiment Setting: {setting}
    Number of Samples: {n_samples}
    Accession: {accession}
    For more information about this study, please check: \n{link}"
  ))
}
