% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DataClasses.R, R/functions-Params.R,
%   R/methods-OnDiskMSnExp.R, R/methods-Params.R
\name{findChromPeaks-centWave}
\alias{findChromPeaks-centWave}
\alias{centWave}
\alias{CentWaveParam}
\alias{findChromPeaks,OnDiskMSnExp,CentWaveParam-method}
\alias{as.list,CentWaveParam-method}
\title{Chromatographic peak detection using the centWave method}
\usage{
CentWaveParam(
  ppm = 25,
  peakwidth = c(20, 50),
  snthresh = 10,
  prefilter = c(3, 100),
  mzCenterFun = "wMean",
  integrate = 1L,
  mzdiff = -0.001,
  fitgauss = FALSE,
  noise = 0,
  verboseColumns = FALSE,
  roiList = list(),
  firstBaselineCheck = TRUE,
  roiScales = numeric(),
  extendLengthMSW = FALSE,
  verboseBetaColumns = FALSE
)

\S4method{findChromPeaks}{OnDiskMSnExp,CentWaveParam}(
  object,
  param,
  BPPARAM = bpparam(),
  return.type = "XCMSnExp",
  msLevel = 1L,
  ...
)

\S4method{as.list}{CentWaveParam}(x, ...)
}
\arguments{
\item{ppm}{\code{numeric(1)} defining the maximal tolerated m/z deviation in
consecutive scans in parts per million (ppm) for the initial ROI
definition.}

\item{peakwidth}{\code{numeric(2)} with the expected approximate
peak width in chromatographic space. Given as a range (min, max)
in seconds.}

\item{snthresh}{\code{numeric(1)} defining the signal to noise ratio cutoff.}

\item{prefilter}{\code{numeric(2)}: \code{c(k, I)} specifying the prefilter
step for the first analysis step (ROI detection). Mass traces are only
retained if they contain at least \code{k} peaks with intensity
\verb{>= I}.}

\item{mzCenterFun}{Name of the function to calculate the m/z center of the
chromatographic peak. Allowed are: \code{"wMean"}: intensity weighted
mean of the peak's m/z values, \code{"mean"}: mean of the peak's m/z
values, \code{"apex"}: use the m/z value at the peak apex,
\code{"wMeanApex3"}: intensity weighted mean of the m/z value at the
peak apex and the m/z values left and right of it and \code{"meanApex3"}:
mean of the m/z value of the peak apex and the m/z values left and right
of it.}

\item{integrate}{Integration method. For \code{integrate = 1} peak limits
are found through descent on the mexican hat filtered data, for
\code{integrate = 2} the descent is done on the real data. The latter
method is more accurate but prone to noise, while the former is more
robust, but less exact.}

\item{mzdiff}{\code{numeric(1)} representing the minimum difference in m/z
dimension required for peaks with overlapping retention times; can be
negative to allow overlap. During peak post-processing, peaks
defined to be overlapping are reduced to the one peak with the largest
signal.}

\item{fitgauss}{\code{logical(1)} whether or not a Gaussian should be fitted
to each peak. This affects mostly the retention time position of the
peak.}

\item{noise}{\code{numeric(1)} allowing to set a minimum intensity required
for centroids to be considered in the first analysis step (centroids with
intensity \verb{< noise} are omitted from ROI detection).}

\item{verboseColumns}{\code{logical(1)} whether additional peak meta data
columns should be returned.}

\item{roiList}{An optional list of regions-of-interest (ROI) representing
detected mass traces. If ROIs are submitted the first analysis step is
omitted and chromatographic peak detection is performed on the submitted
ROIs. Each ROI is expected to have the following elements specified:
\code{scmin} (start scan index), \code{scmax} (end scan index),
\code{mzmin} (minimum m/z), \code{mzmax} (maximum m/z), \code{length}
(number of scans), \code{intensity} (summed intensity). Each ROI should
be represented by a \code{list} of elements or a single row
\code{data.frame}.}

\item{firstBaselineCheck}{\code{logical(1)}. If \code{TRUE} continuous
data within regions of interest is checked to be above the first baseline.
In detail, a first rough estimate of the noise is calculated and peak
detection is performed only in regions in which multiple sequential
signals are higher than this first estimated baseline/noise level.}

\item{roiScales}{Optional numeric vector with length equal to \code{roiList}
defining the scale for each region of interest in \code{roiList} that
should be used for the centWave-wavelets.}

\item{extendLengthMSW}{Option to force centWave to use all scales when
running centWave rather than truncating with the EIC length. Uses the
"open" method to extend the EIC to a integer base-2 length prior to
being passed to \code{convolve} rather than the default "reflect" method. See
https://github.com/sneumann/xcms/issues/445 for more information.}

\item{verboseBetaColumns}{Option to calculate two additional metrics of peak
quality via comparison to an idealized bell curve. Adds \code{beta_cor} and
\code{beta_snr} to the \code{chromPeaks} output, corresponding to a Pearson
correlation coefficient to a bell curve with several degrees of skew
as well as an estimate of signal-to-noise using the residuals from the
best-fitting bell curve. See https://github.com/sneumann/xcms/pull/685
and https://doi.org/10.1186/s12859-023-05533-4 for more information.}

\item{object}{For \code{findChromPeaks()}: an
\code{\link[MSnbase:OnDiskMSnExp-class]{MSnbase::OnDiskMSnExp()}}  object containing the MS- and all
other experiment-relevant data.

\if{html}{\out{<div class="sourceCode">}}\preformatted{For all other methods: a parameter object.
}\if{html}{\out{</div>}}}

\item{param}{An \code{CentWaveParam()} object containing all settings for the
centWave algorithm.}

\item{BPPARAM}{A parameter class specifying if and how parallel processing
should be performed. It defaults to \code{\link[BiocParallel:register]{BiocParallel::bpparam()}}.
See documentation of the \emph{BiocParallel} package for more details. If
parallel processing is enabled, peak detection is performed in parallel
on several of the input samples.}

\item{return.type}{Character specifying what type of object the method should
return. Can be either \code{"XCMSnExp"} (default), \code{"list"} or
\code{"xcmsSet"}.}

\item{msLevel}{\code{integer(1)} defining the MS level on which the peak
detection should be performed. Defaults to \code{msLevel = 1}.}

\item{...}{ignored.}

\item{x}{The parameter object.}
}
\value{
The \code{CentWaveParam()} function returns a \code{CentWaveParam}
class instance with all of the settings specified for chromatographic
peak detection by the centWave method.

For \code{findChromPeaks()}: if \code{return.type = "XCMSnExp"} an
\code{\link[=XCMSnExp]{XCMSnExp()}} object with the results of the peak detection.
If \code{return.type = "list"} a list of length equal to the number of
samples with matrices specifying the identified peaks.
If \code{return.type = "xcmsSet"} an \code{xcmsSet} object
with the results of the peak detection.
}
\description{
The centWave algorithm perform peak density and wavelet based
chromatographic peak detection for high resolution LC/MS data in centroid
mode \emph{Tautenhahn 2008}.

The \verb{findChromPeaks,OnDiskMSnExp,CentWaveParam()} method
performs chromatographic peak detection using the \emph{centWave}
algorithm on all samples from an \code{OnDiskMSnExp}
object. \code{OnDiskMSnExp} objects encapsule all
experiment specific data and load the spectra data (mz and intensity
values) on the fly from the original files applying also all eventual
data manipulations.
}
\details{
The centWave algorithm is most suitable for high resolution
LC/\{TOF,OrbiTrap,FTICR\}-MS data in centroid mode. In the first phase
the method identifies \emph{regions of interest} (ROIs) representing
mass traces that are characterized as regions with less than \code{ppm}
m/z deviation in consecutive scans in the LC/MS map. In detail, starting
with a single m/z, a ROI is extended if a m/z can be found in the next scan
(spectrum) for which the difference to the mean m/z of the ROI is smaller
than the user defined \code{ppm} of the m/z. The mean m/z of the ROI is then
updated considering also the newly included m/z value.

These ROIs are then, after some cleanup, analyzed using continuous wavelet
transform (CWT) to locate chromatographic peaks on different scales.
The first analysis step is skipped, if regions of interest are passed
\emph{via} the \code{param} parameter.

Parallel processing (one process per sample) is supported and can
be configured either by the \code{BPPARAM} parameter or by globally
defining the parallel processing mode using the
\code{\link[BiocParallel:register]{BiocParallel::register()}} method from the \emph{BiocParallel}
package.
}
\note{
These methods and classes are part of the updated and modernized
\emph{xcms} user interface which will eventually replace the
\code{\link[=findPeaks]{findPeaks()}} methods.
}
\examples{

## Create a CentWaveParam object. Note that the noise is set to 10000 to
## speed up the execution of the example - in a real use case the default
## value should be used, or it should be set to a reasonable value.
cwp <- CentWaveParam(ppm = 25, noise = 10000, prefilter = c(3, 10000))
cwp

## Perform the peak detection using centWave on some of the files from the
## faahKO package. Files are read using the `readMsExperiment` function
## from the MsExperiment package
library(faahKO)
library(xcms)
library(MsExperiment)
fls <- dir(system.file("cdf/KO", package = "faahKO"), recursive = TRUE,
           full.names = TRUE)
raw_data <- readMsExperiment(fls[1])

## Perform the peak detection using the settings defined above.
res <- findChromPeaks(raw_data, param = cwp)
head(chromPeaks(res))
}
\references{
Ralf Tautenhahn, Christoph Böttcher, and Steffen Neumann "Highly
sensitive feature detection for high resolution LC/MS" \emph{BMC Bioinformatics}
2008, 9:504
doi: \href{https://doi.org/10.1186/1471-2105-9-504}{10.1186/1471-2105-9-504}
}
\seealso{
The \code{\link[=do_findChromPeaks_centWave]{do_findChromPeaks_centWave()}} core API function and
\code{\link[=findPeaks.centWave]{findPeaks.centWave()}} for the old user interface.

\code{\link[=peaksWithCentWave]{peaksWithCentWave()}} for functions to perform centWave peak
detection in purely chromatographic data.

\code{\link[=XCMSnExp]{XCMSnExp()}} for the object containing the results of
the peak detection.

Other peak detection methods: 
\code{\link{findChromPeaks}()},
\code{\link{findChromPeaks-centWaveWithPredIsoROIs}},
\code{\link{findChromPeaks-massifquant}},
\code{\link{findChromPeaks-matchedFilter}},
\code{\link{findPeaks-MSW}}
}
\author{
Ralf Tautenhahn, Johannes Rainer
}
\concept{peak detection methods}
