% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/do_findChromPeaks-functions.R
\name{do_findChromPeaks_centWave}
\alias{do_findChromPeaks_centWave}
\title{Core API function for centWave peak detection}
\usage{
do_findChromPeaks_centWave(
  mz,
  int,
  scantime,
  valsPerSpect,
  ppm = 25,
  peakwidth = c(20, 50),
  snthresh = 10,
  prefilter = c(3, 100),
  mzCenterFun = "wMean",
  integrate = 1,
  mzdiff = -0.001,
  fitgauss = FALSE,
  noise = 0,
  verboseColumns = FALSE,
  roiList = list(),
  firstBaselineCheck = TRUE,
  roiScales = NULL,
  sleep = 0,
  extendLengthMSW = FALSE,
  verboseBetaColumns = FALSE
)
}
\arguments{
\item{mz}{Numeric vector with the individual m/z values from all scans/
spectra of one file/sample.}

\item{int}{Numeric vector with the individual intensity values from all
scans/spectra of one file/sample.}

\item{scantime}{Numeric vector of length equal to the number of
spectra/scans of the data representing the retention time of each scan.}

\item{valsPerSpect}{Numeric vector with the number of values for each
spectrum.}

\item{ppm}{\code{numeric(1)} defining the maximal tolerated m/z deviation in
consecutive scans in parts per million (ppm) for the initial ROI
definition.}

\item{peakwidth}{\code{numeric(2)} with the expected approximate
peak width in chromatographic space. Given as a range (min, max)
in seconds.}

\item{snthresh}{\code{numeric(1)} defining the signal to noise ratio cutoff.}

\item{prefilter}{\code{numeric(2)}: \code{c(k, I)} specifying the prefilter
step for the first analysis step (ROI detection). Mass traces are only
retained if they contain at least \code{k} peaks with intensity
\verb{>= I}.}

\item{mzCenterFun}{Name of the function to calculate the m/z center of the
chromatographic peak. Allowed are: \code{"wMean"}: intensity weighted
mean of the peak's m/z values, \code{"mean"}: mean of the peak's m/z
values, \code{"apex"}: use the m/z value at the peak apex,
\code{"wMeanApex3"}: intensity weighted mean of the m/z value at the
peak apex and the m/z values left and right of it and \code{"meanApex3"}:
mean of the m/z value of the peak apex and the m/z values left and right
of it.}

\item{integrate}{Integration method. For \code{integrate = 1} peak limits
are found through descent on the mexican hat filtered data, for
\code{integrate = 2} the descent is done on the real data. The latter
method is more accurate but prone to noise, while the former is more
robust, but less exact.}

\item{mzdiff}{\code{numeric(1)} representing the minimum difference in m/z
dimension required for peaks with overlapping retention times; can be
negative to allow overlap. During peak post-processing, peaks
defined to be overlapping are reduced to the one peak with the largest
signal.}

\item{fitgauss}{\code{logical(1)} whether or not a Gaussian should be fitted
to each peak. This affects mostly the retention time position of the
peak.}

\item{noise}{\code{numeric(1)} allowing to set a minimum intensity required
for centroids to be considered in the first analysis step (centroids with
intensity \verb{< noise} are omitted from ROI detection).}

\item{verboseColumns}{\code{logical(1)} whether additional peak meta data
columns should be returned.}

\item{roiList}{An optional list of regions-of-interest (ROI) representing
detected mass traces. If ROIs are submitted the first analysis step is
omitted and chromatographic peak detection is performed on the submitted
ROIs. Each ROI is expected to have the following elements specified:
\code{scmin} (start scan index), \code{scmax} (end scan index),
\code{mzmin} (minimum m/z), \code{mzmax} (maximum m/z), \code{length}
(number of scans), \code{intensity} (summed intensity). Each ROI should
be represented by a \code{list} of elements or a single row
\code{data.frame}.}

\item{firstBaselineCheck}{\code{logical(1)}. If \code{TRUE} continuous
data within regions of interest is checked to be above the first baseline.
In detail, a first rough estimate of the noise is calculated and peak
detection is performed only in regions in which multiple sequential
signals are higher than this first estimated baseline/noise level.}

\item{roiScales}{Optional numeric vector with length equal to \code{roiList}
defining the scale for each region of interest in \code{roiList} that
should be used for the centWave-wavelets.}

\item{sleep}{\code{numeric(1)} defining the number of seconds to wait between
iterations. Defaults to \code{sleep = 0}. If \verb{> 0} a plot is
generated visualizing the identified chromatographic peak. Note: this
argument is for backward compatibility only and will be removed in
future.}

\item{extendLengthMSW}{Option to force centWave to use all scales when
running centWave rather than truncating with the EIC length. Uses the
"open" method to extend the EIC to a integer base-2 length prior to
being passed to \code{convolve} rather than the default "reflect" method. See
https://github.com/sneumann/xcms/issues/445 for more information.}

\item{verboseBetaColumns}{Option to calculate two additional metrics of peak
quality via comparison to an idealized bell curve. Adds \code{beta_cor} and
\code{beta_snr} to the \code{chromPeaks} output, corresponding to a Pearson
correlation coefficient to a bell curve with several degrees of skew
as well as an estimate of signal-to-noise using the residuals from the
best-fitting bell curve. See https://github.com/sneumann/xcms/pull/685
and https://doi.org/10.1186/s12859-023-05533-4 for more information.}
}
\value{
A matrix, each row representing an identified chromatographic peak,
with columns:
\itemize{
\item \code{"mz"}: Intensity weighted mean of m/z values of the peak across scans.
\item \code{"mzmin"}: Minimum m/z of the peak.
\item \code{"mzmax"}: Maximum m/z of the peak.
\item \code{"rt"}: Retention time of the peak's midpoint.
\item \code{"rtmin"}: Minimum retention time of the peak.
\item `"rtmax: Maximum retention time of the peak.
\item \code{"into"}: Integrated (original) intensity of the peak.
\item \code{"intb"}: Per-peak baseline corrected integrated peak intensity.
\item \code{"maxo"}: Maximum intensity of the peak.
\item \code{"sn"}: Signal to noise ratio, defined as \code{(maxo - baseline)/sd},
\code{sd} being the standard deviation of local chromatographic noise.
\item \code{"egauss"}: RMSE of Gaussian fit.
}

Additional columns for \code{verboseColumns = TRUE}:
\itemize{
\item \code{"mu"}: Gaussian parameter mu.
\item \code{"sigma"}: Gaussian parameter sigma.
\item \code{"h"}: Gaussian parameter h.
\item \code{"f"}: Region number of the m/z ROI where the peak was localized.
\item \code{"dppm"}: m/z deviation of mass trace across scans in ppm.
\item \code{"scale"}: Scale on which the peak was localized.
\item \code{"scpos"}: Peak position found by wavelet analysis (scan number).
\item \code{"scmin"}: Left peak limit found by wavelet analysis (scan number).
\item \code{"scmax"}: Right peak limit found by wavelet analysis (scan numer).
}

Additional columns for \code{verboseBetaColumns = TRUE}:
\itemize{
\item \code{"beta_cor"}: Correlation between an "ideal" bell curve and the raw data.
\item \code{"beta_snr"}: Signal-to-noise residuals calculated from the beta_cor fit.
}
}
\description{
This function performs peak density and wavelet based
chromatographic peak detection for high resolution LC/MS data in centroid
mode \emph{Tautenhahn 2008}.
}
\details{
This algorithm is most suitable for high resolution
LC/\{TOF,OrbiTrap,FTICR\}-MS data in centroid mode. In the first phase
the method identifies \emph{regions of interest} (ROIs) representing
mass traces that are characterized as regions with less than \code{ppm}
m/z deviation in consecutive scans in the LC/MS map. In detail, starting
with a single m/z, a ROI is extended if a m/z can be found in the next scan
(spectrum) for which the difference to the mean m/z of the ROI is smaller
than the user defined \code{ppm} of the m/z. The mean m/z of the ROI is then
updated considering also the newly included m/z value.

These ROIs are then, after some cleanup, analyzed using continuous wavelet
transform (CWT) to locate chromatographic peaks on different scales. The
first analysis step is skipped, if regions of interest are passed with
the \code{roiList} parameter.
}
\note{
The \emph{centWave} was designed to work on centroided mode, thus it
is expected that such data is presented to the function.

\if{html}{\out{<div class="sourceCode">}}\preformatted{This function exposes core chromatographic peak detection functionality
of the *centWave* method. While this function can be called
directly, users will generally call the corresponding method for the
data object instead.
}\if{html}{\out{</div>}}
}
\examples{
## Load the test file
faahko_sub <- loadXcmsData("faahko_sub")

## Subset to one file and restrict to a certain retention time range
data <- filterRt(filterFile(faahko_sub, 1), c(2500, 3000))

## Get m/z and intensity values
mzs <- mz(data)
ints <- intensity(data)

## Define the values per spectrum:
valsPerSpect <- lengths(mzs)

## Calling the function. We're using a large value for noise and prefilter
## to speed up the call in the example - in a real use case we would either
## set the value to a reasonable value or use the default value.
res <- do_findChromPeaks_centWave(mz = unlist(mzs), int = unlist(ints),
    scantime = rtime(data), valsPerSpect = valsPerSpect, noise = 10000,
    prefilter = c(3, 10000))
head(res)
}
\references{
Ralf Tautenhahn, Christoph Böttcher, and Steffen Neumann "Highly
sensitive feature detection for high resolution LC/MS"
\emph{BMC Bioinformatics} 2008, 9:504
doi: \href{https://doi.org/10.1186/1471-2105-9-504}{10.1186/1471-2105-9-504}
}
\seealso{
Other core peak detection functions: 
\code{\link{do_findChromPeaks_centWaveWithPredIsoROIs}()},
\code{\link{do_findChromPeaks_massifquant}()},
\code{\link{do_findChromPeaks_matchedFilter}()},
\code{\link{do_findPeaks_MSW}()}
}
\author{
Ralf Tautenhahn, Johannes Rainer
}
\concept{core peak detection functions}
