% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spsServerCollections.R
\name{dynamicFile}
\alias{dynamicFile}
\alias{dynamicFileServer}
\title{Dynamically generate Shiny file selection component based on option}
\usage{
dynamicFile(
  id,
  title = "Select your file:",
  label = "Browse",
  icon = NULL,
  style = "",
  multiple = FALSE,
  buttonType = "primary",
  placeholder = "No file selected",
  mode = spsOption("mode")
)

dynamicFileServer(
  input,
  session,
  id,
  mode = spsOption("mode"),
  roots = c(root = "default")
)
}
\arguments{
\item{id}{element ID, Use \code{ns()} to wrap the id if you are using within a shiny
module, but DO NOT use \code{ns()} to wrap the id on server side}

\item{title}{element title}

\item{label}{upload button label}

\item{icon}{button icon, an object create by \link[shiny:icon]{shiny::icon}}

\item{style}{additional button style, only works for \code{local} mode}

\item{multiple}{bool, are multiple files allowed?}

\item{buttonType}{string, Bootstrap button markup (color).
Default in SPS is 'primary', other valid values include 'info', 'success', 'default',
'warning', 'danger'.}

\item{placeholder}{string, text to display before the file is uploaded}

\item{mode}{string, one of "local" or "server"}

\item{input}{shiny server input}

\item{session}{shiny server session}

\item{roots}{a named character vector, paths where users can reach
on the server, so only required for "server" mode, default is
current directory + all system volumes. You can lock users to a specific path,
so they are not allowed to browse parent folders.
like only current directory: \code{c(current=getwd())}; a temp folder:
\code{c(current=tempdir())}; unlimited: \code{c(shinyFiles::getVolumes()())}}
}
\value{
a Shiny upload component on UI

For the server end it returns a
\strong{reactive} object which is a dataframe, need to extract the value inside reactive
expression, observe, or inside \code{isolate}. See examples
}
\description{
Depending on the "mode" in SPS options, this function renders
a similar UI components but behaves differently on server.
\enumerate{
\item \code{local} mode will not copy file, directly
use a path pointer.
\item \code{server} mode upload file and store in temp. Expect
similar behavior as \link[shiny:fileInput]{shiny::fileInput}.
}
}
\details{
To setup the option:

The \code{local} mode uses functions from \link{shinyFiles} so it will reach file system
on the \strong{server end}. Although the latest \link{shinyFiles} limits users to only
specified server end location (folder), there is still some \strong{risk}. That's why
it is named "local", you are encouraged to run the app on your local computer.
The advantage of "local" is: for some very large files, it does not upload and
store in the temp. Rather, it directly parses the path on the local file system
and return the path immediately. It means the file has to exist on the file
system that serves the Shiny app. If you deploy the app on places like shinyapps.io,
users can only choose files from server.

On the other hand, \code{server} mode uses original but enhanced shiny default upload component.
Users can upload files from local to server. So users do not have access to
server end file system if you deploy it online. However, the limitations are:
\enumerate{
\item not ideal for large files, default limit is 30MB, and there is no break-point
upload.
\item If you are running the app on your own computer, local end and server end is
the same, which is your computer. Using \code{server} mode will make a copy of your
existing file to temp location and this is a waste of time and storage.
}

To set up options:
\enumerate{
\item Under SPS framework, edit options in \code{global.R}.
\item Outside SPS framework with your own Shiny app, use \code{\link[spsUtil:spsOption]{spsUtil::spsOption()}} function,
like \code{spsUtil::spsOption("mode", "server")} or \code{spsUtil::spsOption("mode", "local")} to
set up mode.
}

If you are not sure what mode you are on, use \code{spsUtil::spsOption('mode')} to check.
}
\examples{
# Simple example
if(interactive()){
    spsOption("mode", value = "server") # Change the value to 'local' to see difference
    ui <- fluidPage(
        dynamicFile(id = "server_file", label = "server"),
        verbatimTextOutput("server_out")
    )

    server <- function(input,output,session){
        file_server <- dynamicFileServer(input,session, id = "server_file")
        output$server_out <- renderPrint({
            file_server()  # remember to use `()` for reactive value
        })
    }
    shinyApp(ui = ui, server = server)
}
# To demostrate different modes in the same app, we can set options before the function.
# This is NOT recommended, you should stick with only one mode for the entire app.
if(interactive()){
    spsOption("mode", "local")
    local_ui <- dynamicFile("local_file", "local")
    spsOption("mode", "server")
    server_ui <- dynamicFile("server_file", "server")
    ui <- fluidPage(
        column(
            6,
            local_ui,
            verbatimTextOutput("local_out")
        ),
        column(
            6,
            server_ui,
            verbatimTextOutput("server_out")
        )
    )

    server <- function(input,output,session){
        spsOption("mode", "local")
        file_local <- dynamicFileServer(input,session, id = "local_file")
        output$local_out <- renderPrint({
            file_local() # remember to use `()` for reactive value
        })
        spsOption("mode", "server")
        file_server <- dynamicFileServer(input,session, id = "server_file")
        output$server_out <- renderPrint({
            file_server()
        })
    }
    shinyApp(ui = ui, server = server)
}
}
