% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/funcGam.R
\name{functionalGam}
\alias{functionalGam}
\title{General additive model with functional response}
\usage{
functionalGam(
  data,
  x,
  designmat,
  weights,
  formula,
  family = stats::gaussian(link = "log"),
  ...
)
}
\arguments{
\item{data}{a dataframe with the following columns: Y = functional response;
sample_id = sample ID; image_id = image ID;}

\item{x}{the x-axis values of the functional response}

\item{designmat}{a design matrix as defined by model.matrix()}

\item{weights}{weights as the number of points per image. These weights are
normalised by the mean of the weights in the fitting process}

\item{formula}{the formula for the model. The colnames of the designmatrix
have to correspond to the variables in the formula.}

\item{family}{the distributional family as implemented in \code{family.mgcv}. For
fast computation the default is set to \code{gaussian} with a log link.
other interesting options can be \code{betar} and \code{scat}
\itemize{
\item for more information see \code{family.mgcv}.
}}

\item{...}{Other parameters passed to \code{pffr}}
}
\value{
a fitted pffr object which inherits from gam
}
\description{
A function that takes the output of a metric calculation as done by
\code{calcMetricPerFov}. The data has to be prepared into the correct format for
the functional analysis by the \code{prepData} function. The output is a \code{pffr}
object as implemented by \code{refund}.
}
\examples{
# load the pancreas dataset
library("tidyr")
library("dplyr")
# retrieve example data from Damond et al. (2019)
spe <- .loadExample()
# calculate the Gcross metric for alpha and Tc cells
metricRes <- calcMetricPerFov(spe, c("alpha", "Tc"),
    subsetby = "image_number", fun = "Gcross",
    marks = "cell_type", rSeq = seq(0, 50, length.out = 50),
    c("patient_stage", "patient_id", "image_number"), ncores = 1
)
metricRes$ID <- paste0(
  metricRes$patient_stage, "|", metricRes$patient_id,
  "|", metricRes$image_number
)
dat <- prepData(metricRes, "r", "rs", sample_id = "patient_id",
    image_id = "image_number", condition = "patient_stage")

#' # drop rows with NA
dat <- dat |> drop_na()

# create a designmatrix
condition <- dat$patient_stage
# relevel the condition - can set explicit contrasts here
condition <- relevel(condition, "Non-diabetic")
designmat <- model.matrix(~condition)
# colnames don't work with the '-' sign
colnames(designmat) <- c(
    "(Intercept)", "conditionLong_duration",
    "conditionOnset"
)
# fit the model
mdl <- functionalGam(
    data = dat, x = metricRes$r |> unique(),
    designmat = designmat, weights = dat$npoints,
    formula = formula(Y ~ conditionLong_duration +
        conditionOnset + s(patient_id, bs = "re")),
        algorithm = "bam"
)
summary(mdl)
}
