% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scHiC_bulk_compare.R
\name{scHiC_bulk_compare}
\alias{scHiC_bulk_compare}
\title{Compare Bulk Hi-C Data}
\usage{
scHiC_bulk_compare(
  norm.hic.table,
  D.interval,
  fprControl.logfc = 0.8,
  alpha = 0.05,
  SD = 2,
  numChanges = NULL,
  FC = 3,
  A.min = NULL,
  Plot = TRUE,
  BPPARAM = bpparam()
)
}
\arguments{
\item{norm.hic.table}{A data frame representing a jointly normalized
pseudo-bulk Hi-C table output from two conditions,
generated by the `hic_loess()` function. This should be a pre-processed
table that has been jointly normalized before differential analysis.}

\item{D.interval}{A numeric vector defining the distance intervals to
consider in the analysis, or a character string 'full' indicating the
inclusion of all genomic distances in the analysis. Genomic distance refers
to the number of base pairs between two regions in the genome (e.g., loci
or bins) that is scaled by resolution D = (start2 - start1)/resolution
(e.g., D = (16,000,000 - 17,000,000)/1,000,000 -> D = 1)}

\item{fprControl.logfc}{A numeric value controlling the false positive rate
by setting the threshold for the log fold change in the 'difference'
cluster. Detected differences identified by Gaussian Mixed Model (GMM)
clusters only include values that are larger than this threshold.
Default is 0.8.}

\item{alpha}{A numeric value for the significance level of outlier detection
in the `hic_compare()` function from HiCcompare. The default is 0.05.}

\item{SD}{A numeric value specifying the standard deviation threshold for
fuzzing, used to produce a simulated Hi-C matrix. This value is used to
modify the process of finding the optimal 'A.min' quantile cutoff for
detecting significant outliers. Users can select the value based on their
assumption of the scHi-C data. The default is 2.}

\item{numChanges}{An integer or NULL, indicating the number of changes to add
 to the simulated Hi-C matrix. This value is used to modify the process of
 finding the optimal 'A.min' quantile cutoff for detecting significant
 outliers. Based on the user's assumption about the possible number of
 differences, they can set the number of changes that should be proportional
 to the resolution of the data. High-resolution data should be assumed to
 have more changes. If `numChanges` = NULL, the function is set to the
 number of changes (or simulated difference) scaled by a factor of 30
 (e.g., 1MB resolution - 30 changes, 500KB resolution - 60 changes, etc.)
The default is NULL.}

\item{FC}{A numeric value representing the fold change threshold added to
the simulated Hi-C matrix. This value is used to identify the optimal
'A.min' quantile cutoff for detecting significant outliers. Users can
select the FC value based on their assumption of difference fold change in
their data. The default is 3.}

\item{A.min}{A numeric value or NULL, specifying the A-value quantile cutoff
to filter lower average expression in the `hic_compare()` function from
HiCcompare. `hic_compare()` is used to detect outliers, which is assumed to
be 'differences' bins in case of its number is too small (or none) to be
clustered by GMM method. If not provided, an optimized minimum A threshold
that maximizes MCC and TPR while minimizing FPR in the simulated Hi-C
matrix.}

\item{Plot}{A logical value indicating whether to plot the differential
results in an MD plot. The default is TRUE.}

\item{BPPARAM}{Parameters for `BiocParallel`, to be passed to the `bpparam()`
function. See `?bpparam()` for more info.}
}
\value{
A data frame containing the results of the differential Hi-C
 analysis, including a normalized report and a 'Difference.cluster' column
 indicating the clusters of differences identified in the analysis.
}
\description{
This function compares single-cell Hi-C data between two groups using the
`scHicCompare` differential analysis workflow. It detects chromatin
interaction differences between the single-cell Hi-C data of two cell types
or conditions by Gaussian Mixture Model (GMM) cluster and outlier
differences by `hic_compare()` analysis of `HiCcompare`
}
\details{
The `scHiC_bulk_compare()` function performs differential chromatin
 interaction comparisons between single-cell Hi-C data from two cell type
 groups or conditions. The workflow includes clustering normalized log fold
 changes between interaction frequencies of 2 cell types at each genomic
 distance into "difference" and "non-difference" groups. The non-difference
 group is assumed to follow a normal distribution centered around 0 and is
 clustered by a Gaussian Mixture Model. The difference cluster consists of
 points belonging to other distributions. In cases where the size of the
 differences is not large enough to form distinct distributions, these
 differences are assumed to be outliers of the normal distribution,
 which are identified by the `hic_compare()` function.
}
\examples{
# Load data folder example to the current working directory
ODCs_example <- system.file("extdata/ODCs_example", package = "scHiCcompare")
MGs_example <- system.file("extdata/MGs_example", package = "scHiCcompare")
# Input single-cell Hi-C in sparse format (.txt) from a path
scHiC.table_ODC <- scHiC_table(
  file.path = ODCs_example,
  cell.type = "ODC",
  select.chromosome = "chr20"
)
scHiC.table_MG <- scHiC_table(
  file.path = MGs_example,
  cell.type = "MG",
  select.chromosome = "chr20"
)
# Bulk matrix in sparse format
bulk.sparse.1 <- na.omit(pseudo_bulkHic(
  scHiC.table = scHiC.table_ODC,
  out = "sparse"
))
bulk.sparse.2 <- na.omit(pseudo_bulkHic(
  scHiC.table = scHiC.table_MG,
  out = "sparse"
))
# Create the `hic.table` object
library(HiCcompare)
bulk.hic.table <- create.hic.table(bulk.sparse.1, bulk.sparse.2,
  chr = "chr20", scale = FALSE
)
# Jointly normalize data for a single chromosome
hic.table_normalize <- hic_loess(bulk.hic.table,
  Plot = TRUE,
  Plot.smooth = FALSE
)
# Example usage of the BulkHiC_compare function
result <- scHiC_bulk_compare(hic.table_normalize,
  D.interval = c(1:10),
  fprControl.logfc = 0.8
)

}
\references{
Stansfield JC, Cresswell KG, Vladimirov VI  et al (2018). Hiccompare: an
R-package for joint normalization and comparison of HI-C datasets.
BMC Bioinformatics  2018;19:279.

Scrucca L., Fop M., Murphy T. B. and Raftery A. E. (2016) mclust 5:
clustering, classification and density estimation using Gaussian finite
mixture models, The R Journal, 8/1, pp. 289-317.

C. Fraley and A. E. Raftery (2007) Bayesian regularization for normal
mixture estimation and model-based clustering.
Journal of Classification, 24, 155-181.

Fraley C. and Raftery A. E. (2002) Model-based clustering, discriminant
analysis and density estimation, Journal of the American Statistical
Association, 97/458, pp. 611-631.

Patrick Royston (1982). Algorithm AS 181: The W test for Normality.
Applied Statistics, 31, 176–180. doi:10.2307/2347986.

Patrick Royston (1982). An extension of Shapiro and Wilk's W test for
normality to large samples. Applied Statistics, 31, 115–124.
doi:10.2307/2347973.
}
