% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_consensus.R
\name{create_consensus}
\alias{create_consensus}
\title{Create a Consensus Adjacency Matrix from Multiple Networks}
\usage{
create_consensus(
  adj_matrix_list,
  method = c("vote", "union", "INet"),
  weighted_list = NULL,
  theta = 0.04,
  threshold = 0.5,
  ncores = 1,
  tolerance = 0.1,
  nitermax = 50,
  verbose = FALSE
)
}
\arguments{
\item{adj_matrix_list}{A \linkS4class{SummarizedExperiment} object
containing binary adjacency matrices (square, 0/1) with identical
dimensions and matching row/column names, or a list of such matrices.}

\item{method}{Character string specifying the consensus strategy. One of:
\itemize{
\item \code{"vote"} (default): An edge is included if supported
by at least \code{threshold} fraction of matrices.
\item \code{"union"}: An edge is included if present in any
matrix.
\item \code{"INet"}: Combines normalized weighted matrices using
\code{\link[INetTool]{consensusNet}}.
}}

\item{weighted_list}{A \linkS4class{SummarizedExperiment} object containing
weighted adjacency matrices (required if \code{method = "INet"}), or a
list of such matrices.}

\item{theta}{Numeric. Tuning parameter passed to \code{consensusNet}
(default: \code{0.04}).}

\item{threshold}{Numeric between 0 and 1. Threshold for "vote" and
"INet" methods. Default is \code{0.5}.}

\item{ncores}{Integer. Number of CPU cores to use when \code{method =
"INet"}. Default is \code{1}.}

\item{tolerance}{Numeric. Tolerance for differences between similar graphs
in INet method. Default is \code{0.1}.}

\item{nitermax}{Integer. Maximum number of iterations for INet algorithm.
Default is \code{50}.}

\item{verbose}{Logical. If TRUE, display verbose output for INet method.
Default is \code{FALSE}.}
}
\value{
A \linkS4class{SummarizedExperiment} object with a single assay
containing the consensus adjacency matrix (binary or weighted, depending
on the method). Metadata includes consensus method and parameters.
}
\description{
Builds a consensus adjacency matrix from networks stored in a
\linkS4class{SummarizedExperiment} using one of three methods:
\code{"vote"}, \code{"union"}, or \code{"INet"}.
}
\details{
Consensus construction depends on the selected method:
\describe{
\item{\strong{vote}}{Counts the presence of each edge across all
matrices and includes edges supported by at least
\code{threshold × N} matrices.}
\item{\strong{union}}{Includes any edge that appears in any
matrix.}
\item{\strong{INet}}{Multiplies binary matrices by corresponding
weighted matrices, normalizes the results, and applies
\code{consensusNet} to generate a consensus network.}
}

For "INet", both binary and weighted adjacency matrices must be
provided with matching dimensions.
}
\examples{
data(toy_counts)


# Infer networks (toy_counts is already a MultiAssayExperiment)
networks <- infer_networks(
    count_matrices_list = toy_counts,
    method = "GENIE3",
    nCores = 1
)
head(networks[[1]])

# Generate adjacency matrices
wadj_se <- generate_adjacency(networks)
swadj_se <- symmetrize(wadj_se, weight_function = "mean")

# Apply cutoff
binary_se <- cutoff_adjacency(
    count_matrices = toy_counts,
    weighted_adjm_list = swadj_se,
    n = 1,
    method = "GENIE3",
    quantile_threshold = 0.95,
    nCores = 1,
    debug = TRUE
)
head(binary_se[[1]])

consensus <- create_consensus(binary_se, method = "union")
head(consensus)
}
