% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/region_count_functions.R
\name{get_region_counts}
\alias{get_region_counts}
\title{Retrieves the region counts from a .ribo file}
\usage{
get_region_counts(
  ribo.object,
  range.lower = length_min(ribo.object),
  range.upper = length_max(ribo.object),
  length = TRUE,
  transcript = TRUE,
  tidy = TRUE,
  alias = FALSE,
  normalize = FALSE,
  region = c("UTR5", "UTR5J", "CDS", "UTR3J", "UTR3"),
  compact = TRUE,
  experiment = experiments(ribo.object)
)
}
\arguments{
\item{ribo.object}{A 'Ribo' object}

\item{range.lower}{Lower bound of the read length, inclusive}

\item{range.upper}{Upper bound of the read length, inclusive}

\item{length}{Logical value that denotes if the region count information should be summed across read lengths}

\item{transcript}{Logical value that denotes if the region count information should be summed across transcripts}

\item{tidy}{Option to return the data frame in a tidy format}

\item{alias}{Option to report the transcripts as aliases/nicknames}

\item{normalize}{Option to normalize the counts as counts per million reads}

\item{region}{Specific region of interest}

\item{compact}{Option to return a DataFrame with Rle and factor as opposed to a raw data.frame}

\item{experiment}{List of experiment names}
}
\value{
An annotated DataFrame or data.frame (if the compact parameter is set to FALSE) of the
region count information for the regions specified in the 'region' parameter. The returned data frame
will have a length column when the 'length' parameter is set to FALSE, indicating that the count
information will not be summed across the provided range of read lengths. Similarly, the returned data
frame will have a transcript column when the 'transcript' parameter is set to FALSE, indicating that the
count information will not be summed across the transcripts. In the case that transcript parameter is
'FALSE', the returned data frame will present the transcripts according to the aliases specified at the
creation of the ribo object if the 'alias' parameter is set to TRUE.
}
\description{
\code{\link{get_region_counts}} will return the particular region counts
of any subset of regions for a given set of experiments.
}
\details{
This function will return a data frane of the counts at each specified region
for each specified experiment. The region options are "UTR5", "UTR5J", "CDS",
"UTR3J", and "UTR3". The user can specify any subset of regions in the form of a vector,
a list, or a single string if only one region is desired.

The dimensions of the returned DataFrame depend on the parameters
range.lower, range.upper, length, and transcript.

The param 'length' condenses the read lengths together.
When length is TRUE and transcript is FALSE, the
data frame presents information for each transcript across
all of the read lengths. That is, each transcript has a value
that is the sum of all of the counts across every read length.
As a result, information about the transcript at each specific
read length is lost.

The param 'transcript' condenses the transcripts together.
When transcript is TRUE and length is FALSE data
frame presents information at each read length between range.lower and
range.upper inclusive. That is, each separate read length denotes the
sum of counts from every transcript. As a result, information about the
counts of each individual transcript is lost.

When 'transcript' is set to FALSE, the 'alias' parameter specifies whether
or not the returned DataFrame should present each transcript as an alias
instead of the original name. If 'alias' is set to TRUE, then the column
of the transcript names will contain the aliases rather than the original
reference names of the .ribo file.

If both 'length' and 'transcript' are TRUE, then the resulting
DataFrame prints out one row for each experiment. This provides the metagene
information across all transcripts and all reads in a given experiment.

If both length' and 'transcript' are FALSE, calculations are done to the data,
all information is preserved for both the read length and the transcript.
The DataFrame would just present the entire stored raw data
from the read length 'range.lower' to the read length 'range.upper' which in most
cases would result in a slow run time with a massive DataFrame returned.

When 'transcript' is set to FALSE, the 'alias' parameter specifies whether
or not the returned DataFrame should present each transcript as an alias
instead of the original name. If 'alias' is set to TRUE, then the column
of the transcript names will contain the aliases rather than the original
reference names of the .ribo file.
}
\examples{
#generate the ribo object
file.path <- system.file("extdata", "sample.ribo", package = "ribor")
sample <- Ribo(file.path)

#specify the regions and experiments of interest
regions <- c("UTR5", "UTR5J", "CDS", "UTR3J", "UTR3")
experiments <- c("Hela_1", "Hela_2", "WT_1")

#obtains the region counts at each individual read length, summed across every transcript
region.counts <- get_region_counts(ribo.object = sample,
                                   region = regions,
                                   range.lower = 2,
                                   range.upper = 5,
                                   length = FALSE,
                                   transcript = TRUE,
                                   tidy = FALSE,
                                   alias = FALSE,
                                   experiment = experiments)

}
