% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/renderReport.R
\docType{data}
\name{renderReport}
\alias{renderReport}
\alias{templatePvalueDensity}
\alias{templateDensity}
\alias{templateManhattan}
\alias{templatePvalueHistogram}
\alias{templateHistogram}
\title{Generate a HTML/PDF report exploring a set of genomic regions}
\format{
An object of class \code{character} of length 1.

An object of class \code{character} of length 1.

An object of class \code{character} of length 1.

An object of class \code{character} of length 1.

An object of class \code{character} of length 1.
}
\usage{
renderReport(
  regions,
  project = "",
  pvalueVars = c(`P-values` = "pval"),
  densityVars = NULL,
  significantVar = mcols(regions)$pval <= 0.05,
  annotation = NULL,
  nBestRegions = 500,
  customCode = NULL,
  outdir = "regionExploration",
  output = "regionExploration",
  browse = interactive(),
  txdb = NULL,
  device = "png",
  densityTemplates = list(Pvalue = templatePvalueDensity, Common = templateDensity,
    Manhattan = templateManhattan),
  template = NULL,
  theme = NULL,
  digits = 2,
  ...
)

templatePvalueDensity

templateDensity

templateManhattan

templatePvalueHistogram

templateHistogram
}
\arguments{
\item{regions}{The set of genomic regions of interest as a \code{GRanges}
object. All sequence lengths must be provided.}

\item{project}{The title of the project.}

\item{pvalueVars}{The names of the variables with values between 0 and 1 to
plot density values by chromosome and a table for commonly used cutoffs.
Most commonly used to explore p-value distributions. If a named
character vector is provided, the names are used in the plot titles.}

\item{densityVars}{The names of variables to use for making density plots
by chromosome. Commonly used to explore scores and other variables given
by region.  If a named character vector is provided, the names are used in
the plot titles.}

\item{significantVar}{A \code{logical} variable differentiating statistically
significant regions from the rest. When provided, both types of regions
are compared against each other to see differences in width, location, etc.}

\item{annotation}{The output from \link[bumphunter]{matchGenes} used on
\code{regions}. Note that this can take time for a large set of regions
so it's better to pre-compute this information and save it.}

\item{nBestRegions}{The number of regions to include in the interactive
table.}

\item{customCode}{An absolute path to a child R Markdown file with code to be
evaluated before the reproducibility section. Its useful for users who want
to customize the report by adding conclusions derived from the data and/or
further quality checks and plots.}

\item{outdir}{The name of output directory.}

\item{output}{The name of output HTML file (without the html extension).}

\item{browse}{If \code{TRUE} the HTML report is opened in your browser once
it's completed.}

\item{txdb}{Specify the transcription database to use for identifying the
closest genes via \link[bumphunter]{matchGenes}. If \code{NULL} it will
use TxDb.Hsapiens.UCSC.hg19.knownGene by default.}

\item{device}{The graphical device used when knitting. See more at
\url{http://yihui.name/knitr/options} (\code{dev} argument).}

\item{densityTemplates}{A list of length 3 with templates for the p-value
density plots (variables from \code{pvalueVars}), the continuous
variables density plots (variables from \code{densityVars}), and Manhattan
plots for the p-value variables (\code{pvalueVars}). These templates
are processed by \link[whisker]{whisker.render}. Check the default templates
for more information. The \code{densityTemplates} argument is available for
those users interested in customizing these plots. For example, to show
histograms instead of density plots use \code{templatePvalueHistogram} and
\code{templateHistogram} instead of \code{templatePvalueDensity} and
\code{templateDensity} respectively.}

\item{template}{Template file to use for the report. If not provided, will
use the default file found in regionExploration/regionExploration.Rmd
within the package source.}

\item{theme}{A ggplot2 \link[ggplot2]{theme} to use for the plots made with
ggplot2.}

\item{digits}{The number of digits to round to in the interactive table of
the top \code{nBestRegions}. Note that p-values and adjusted p-values won't
be rounded.}

\item{...}{Arguments passed to other methods and/or advanced arguments.
Advanced arguments:
\describe{
\item{overviewParams }{ A two element list with \code{base_size} and
\code{areaRel} that control the text size for the genomic overview plots.}
\item{output_format }{ Either \code{html_document}, \code{pdf_document} or
\code{knitrBootstrap::bootstrap_document} unless you modify the YAML
template.}
\item{clean }{ Logical, whether to clean the results or not. Passed to
\link[rmarkdown]{render}.}
}}
}
\value{
An HTML report with a basic exploration for the given set of
genomic regions. See the example report at
\url{http://leekgroup.github.io/regionReport/reference/renderReport-example/regionExploration.html}.
}
\description{
This function generates a HTML report with quality checks, genome location
exploration, and an interactive table with the results. Other output formats
are possible such as PDF but lose the interactivity. Users can easily append
to the report by providing a R Markdown file to \code{customCode}, or can
customize the entire template by providing an R Markdown file to
\code{template}.
}
\details{
Set \code{output_format} to \code{'knitrBootstrap::bootstrap_document'} or
\code{'pdf_document'} if you want a HTML report styled by knitrBootstrap or
a PDF report respectively. If using knitrBootstrap, we recommend the version
available only via GitHub at \url{https://github.com/jimhester/knitrBootstrap}
which has nicer features than the current version available via CRAN. You can
also set the \code{output_format} to \code{'html_document'} for a HTML
report styled by rmarkdown. The default is set to
\code{'BiocStyle::html_document'}.

If you modify the YAML front matter of \code{template}, you can use other
values for \code{output_format}.

The HTML report styled with knitrBootstrap can be smaller in size than the
\code{'html_document'} report.
}
\examples{

## Load derfinder for an example set of regions
library("derfinder")
regions <- genomeRegions$regions

## Assign chr length
library("GenomicRanges")
seqlengths(regions) <- c("chr21" = 48129895)

## The output will be saved in the 'renderReport-example' directory
dir.create("renderReport-example", showWarnings = FALSE, recursive = TRUE)

## Generate the HTML report
report <- renderReport(regions, "Example run",
    pvalueVars = c(
        "Q-values" = "qvalues", "P-values" = "pvalues"
    ), densityVars = c(
        "Area" = "area", "Mean coverage" = "meanCoverage"
    ),
    significantVar = regions$qvalues <= 0.05, nBestRegions = 20,
    outdir = "renderReport-example"
)

if (interactive()) {
    ## Browse the report
    browseURL(report)
}

## See the example report at
## http://leekgroup.github.io/regionReport/reference/renderReport-example/regionExploration.html


## Check the default templates. For users interested in customizing these
## plots.
## For p-value variables:
cat(regionReport::templatePvalueDensity)

## For continous variables:
cat(regionReport::templateDensity)

## For Manhattan plots
cat(regionReport::templateManhattan)

##################################################
## bumphunter example mentioned in the vignette ##
##################################################

## Load bumphunter
library("bumphunter")

## Create data from the vignette
pos <- list(
    pos1 = seq(1, 1000, 35),
    pos2 = seq(2001, 3000, 35),
    pos3 = seq(1, 1000, 50)
)
chr <- rep(paste0("chr", c(1, 1, 2)), times = sapply(pos, length))
pos <- unlist(pos, use.names = FALSE)

## Find clusters
cl <- clusterMaker(chr, pos, maxGap = 300)

## Build simulated bumps
Indexes <- split(seq_along(cl), cl)
beta1 <- rep(0, length(pos))
for (i in seq(along = Indexes)) {
    ind <- Indexes[[i]]
    x <- pos[ind]
    z <- scale(x, median(x), max(x) / 12)
    beta1[ind] <- i * (-1)^(i + 1) * pmax(1 - abs(z)^3, 0)^3 ## multiply by i to vary size
}

## Build data
beta0 <- 3 * sin(2 * pi * pos / 720)
X <- cbind(rep(1, 20), rep(c(0, 1), each = 10))
set.seed(23852577)
error <- matrix(rnorm(20 * length(beta1), 0, 1), ncol = 20)
y <- t(X[, 1]) \%x\% beta0 + t(X[, 2]) \%x\% beta1 + error

## Perform bumphunting
tab <- bumphunter(y, X, chr, pos, cl, cutoff = .5)

## Explore data
lapply(tab, head)

library("GenomicRanges")

## Build GRanges with sequence lengths
regions <- GRanges(
    seqnames = tab$table$chr,
    IRanges(start = tab$table$start, end = tab$table$end),
    strand = "*", value = tab$table$value, area = tab$table$area,
    cluster = tab$table$cluster, L = tab$table$L, clusterL = tab$table$clusterL
)

## Assign chr lengths
library(GenomeInfoDb)  # for getChromInfoFromUCSC()
seqlengths(regions) <- seqlengths(
    getChromInfoFromUCSC("hg19", as.Seqinfo = TRUE)
)[
    names(seqlengths(regions))
]

## Explore the regions
regions

## Now create the report
report <- renderReport(regions, "Example bumphunter",
    pvalueVars = NULL,
    densityVars = c(
        "Area" = "area", "Value" = "value",
        "Cluster Length" = "clusterL"
    ), significantVar = NULL,
    output = "bumphunter-example", outdir = "bumphunter-example",
    device = "png"
)

## See the example report at
## http://leekgroup.github.io/regionReport/reference/bumphunter-example/bumphunter-example.html
}
\author{
Leonardo Collado-Torres
}
\keyword{datasets}
