% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rawrr.R
\name{readChromatogram}
\alias{readChromatogram}
\title{Extracts chromatographic data from a raw file.}
\usage{
readChromatogram(
  rawfile,
  mass = NULL,
  tol = 10,
  filter = "ms",
  type = "xic",
  tmpdir = tempdir()
)
}
\arguments{
\item{rawfile}{the name of the raw file containing the mass spectrometry data from the Thermo Fisher Scientific instrument.}

\item{mass}{a vector of mass values iff \code{type = 'xic'}.}

\item{tol}{mass tolerance in ppm iff \code{type = 'xic'}.}

\item{filter}{defines the scan filter, default is \code{filter="ms"} if a
wrong filter is set the function will return \code{NULL} and draws a warning.}

\item{type}{\code{c(xic, bpc, tic)} for extracted ion , base peak or
total ion chromatogram.}

\item{tmpdir}{defines the directory used to store temporary data generated by
the .NET assembly \code{rawrr.exe}. The default uses the output of
\code{tempdir()}.}
}
\value{
chromatogram object(s) containing of a vector of \code{times} and a
corresponding vector of \code{intensities}.
}
\description{
Extracts chromatographic data from a raw file.
}
\details{
Chromatograms come in different flavors but are always signal intensity
values as a function of time. Signal intensities can be point estimates from
scanning detectors or plain intensities from non-scanning detectors, e.g.,
UV trace. Scanning detector (mass analyzers) point estimates can be defined
in different ways by, for instance, summing all signals of a given spectrum
(total ion chromatogram or TIC), or by extracting signal around an expected
value (extracted ion chromatogram = XIC), or by using the maximum signal
contained in a spectrum (base peak chromatogram = BPC). On top, chromatograms
can be computed from pre-filtered lists of scans. A total ion chromatogram
(TIC), for instance, is typically generated by iterating over all MS1-level
scans.
}
\examples{

# Example 1: not meaningful but proof-of-concept
(rawfile <- rawrr::sampleFilePath())

rawrr::readChromatogram(rawfile, mass=c(669.8381, 726.8357), tol=1000) |>
    plot()
rawrr::readChromatogram(rawfile, type='bpc') |> plot()
rawrr::readChromatogram(rawfile, type='tic') |> plot()

# Example 2: extract iRT peptides
 if (require(ExperimentHub) & require(protViz)){
iRTpeptide <- c("LGGNEQVTR", "YILAGVENSK", "GTFIIDPGGVIR", "GTFIIDPAAVIR",
  "GAGSSEPVTGLDAK", "TPVISGGPYEYR", "VEATFGVDESNAK",
  "TPVITGAPYEYR", "DGLDAASYYAPVR", "ADVTPADFSEWSK",
  "LFLQFGAQGSPFLK")


# fetch via ExperimentHub
library(ExperimentHub)
eh <- ExperimentHub::ExperimentHub()
EH4547 <- normalizePath(eh[["EH4547"]])

(rawfile <- paste0(EH4547, ".raw"))
if (!file.exists(rawfile)){
    file.link(EH4547, rawfile)
}
op <- par(mfrow=c(2,1))
readChromatogram(rawfile, type='bpc') |> plot()
readChromatogram(rawfile, type='tic') |> plot()
par(op)

# derive [2H+] ions
((protViz::parentIonMass(iRTpeptide) + 1.008) / 2) |>
   readChromatogram(rawfile=rawfile) |>
   plot()
}
}
\references{
Automated quality control sample 1 (autoQC01) analyzed across different
Thermo Scientific mass spectrometers,
\href{https://massive.ucsd.edu/ProteoSAFe/dataset.jsp?accession=MSV000086542}{MSV000086542}.
}
\seealso{
\itemize{
\item{The Thermo Fisher Scientific RawFileReader C# code snippets
\url{https://planetorbitrap.com/rawfilereader}.}
\item{\url{https://CRAN.R-project.org/package=protViz}}
\item{\url{https://massive.ucsd.edu/ProteoSAFe/dataset.jsp?accession=MSV000086542}}
}
}
\author{
Christian Trachsel, Tobias Kockmann and
Christian Panse <cp@fgz.ethz.ch> 2018, 2019, 2020.
}
