context("IO test")

test_that("multi data segment", {
  dataPath <- "~/rglab/workspace/flowCore/misc/"
  filename  <- file.path(dataPath, "multi-datasegment.fcs")
  skip_if_not(file.exists(filename))
  expect_warning(fr <- read.ncdfFlowSet(filename)[[1]], "39 additional data")
  expect_is(fr, "flowFrame")
  expect_equal(nrow(fr), 1244)
  
  fr <- read.ncdfFlowSet(filename, dataset = 1)[[1]]
  expect_equal(nrow(fr), 1244)
  expect_equal(colnames(fr)[10], "FL2-PL")
  
  fr <- read.ncdfFlowSet(filename, dataset = 10)[[1]]
  expect_equal(nrow(fr), 955)
  expect_equal(colnames(fr)[10], "testPL")
})

test_that("read.ncdfFlowSet", {
  
  path <- system.file("extdata","compdata","data",package="flowCore")
  files <- list.files(path,full.names=TRUE)[1:3]
  
  #test alter.names option
  suppressMessages(nc1  <- read.ncdfFlowSet(files=files,ncdfFile="ncfsTest.nc", alter.names = TRUE))
  fs1  <- read.flowSet(files=files, alter.names = TRUE)
  is_equal_flowSet(nc1, fs1)
  expect_equal(getFileName(nc1), "ncfsTest.nc")
  
  #create ncdfFlowSet from fcs with the actual raw data written in cdf
  suppressMessages(nc1  <- read.ncdfFlowSet(files=files,ncdfFile="ncfsTest.nc"))
  fs1  <- read.flowSet(files=files)
  is_equal_flowSet(nc1, fs1)
  expect_equal(getFileName(nc1), "ncfsTest.nc")
  
  
  
  #create empty ncdfFlowSet from fcs and add data slices afterwards
  suppressMessages(nc1  <- read.ncdfFlowSet(files=files,ncdfFile="ncfsTest.nc",isWriteSlice= FALSE))
  expect_equal(nrow(nc1[[1]]), 0)  
  suppressMessages(nc1[[1]] <- fs1[[1]])
  expect_equal(nrow(nc1[[1]]), 1e4)
  expect_equal(nrow(nc1[[2]]), 0)
  unlink(nc1)
  
  #test double delimiter issue
  fr <- read.FCS(files[1])
  tmp <- tempfile()
  write.FCS(fr, tmp, delimiter = "/")
  expect_error(read.ncdfFlowSet(tmp), "Empty keyword name")
  nc1 <- read.ncdfFlowSet(tmp, emptyValue = FALSE)
  expect_equal(exprs(fr), exprs(nc1[[1]]), tolerance = 3e-08, check.attributes = F)
  unlink(nc1)
  
  #test channels argument
  chnls <- colnames(fr)[c(1,3,5)]
  suppressMessages(nc1  <- read.ncdfFlowSet(files=files,channels = chnls))
  is_equal_flowFrame(nc1[[1]], fr[, chnls], description = FALSE)
})


test_that("save_ncfs/load_ncfs", {
      nc <- ncdfFlowSet(GvHD[1:2])
      
      output <- tempfile(pattern = "ncfs")
      
      #save to a new folder
      expect_message(save_ncfs(nc, path = output), "Done")
      expect_message(nc1 <- load_ncfs(output), "Done")
      is_equal_flowSet(nc, nc1)
      
      #overwrite exsiting folder
      expect_error(save_ncfs(nc, path = output), "already exists")
      expect_message(save_ncfs(nc, path = output, overwrite = T), "Done")
      
      #save to the non-existing folder
      suppressWarnings(expect_error(save_ncfs(nc, path = "/faked/folder")))
      
      rdsFile <- list.files(output, pattern = ".rds")
      cdfFile <- list.files(output, pattern = ".nc")
      
      #invalid folder
      newRDS <- "tmp.rds"
      file.rename(file.path(output, rdsFile), file.path(output, newRDS))
      expect_error(save_ncfs(nc, path = output, overwrite = T), "doesn't match")
      
      newRDS1 <- "tmp.dd"
      file.rename(file.path(output, newRDS), file.path(output, newRDS1))
      expect_error(save_ncfs(nc, path = output, overwrite = T), "Not a valid")
      
      #restore rds
      file.rename(file.path(output, newRDS1), file.path(output, rdsFile))
      
      newCDF <- "tmp.nc"
      file.copy(file.path(output, cdfFile), file.path(output, newCDF))
      expect_error(save_ncfs(nc, path = output, overwrite = T), "Not a valid")
      
      #remove the redundant nc file
      file.remove(file.path(output, newCDF))
      
      #remove rds file
      file.remove(file.path(output, rdsFile))
      expect_error(load_ncfs(output), "rds file missing")
      #restore rds file
      saveRDS(nc, file.path(output, rdsFile))
      #remove cdf file
      file.remove(file.path(output, cdfFile))
      #restore it
      file.copy(nc@file, file.path(output, cdfFile))
      #nc.trans
      newCDF <- "tmp.nc.trans"
      file.rename(file.path(output, cdfFile), file.path(output, newCDF))
      expect_message(load_ncfs(output), "Done")
      
      #test the nc generated by read.ncdfFlowSet
      nc <- suppressMessages(read.ncdfFlowSet(list.files(system.file("extdata", package = "flowCore"), full.names = TRUE)[1:2]))
      output <- tempfile(pattern = "ncfs")
      #save to a new folder
      expect_message(save_ncfs(nc, path = output), "Done")
      expect_message(nc1 <- load_ncfs(output), "Done")
      is_equal_flowSet(nc, nc1)
      
      #transform and save
      output <- tempfile(pattern = "ncfs")
      nc.trans <- suppressMessages(transform(nc, transformList("FL1-H", logicleTransform())))
      expect_message(save_ncfs(nc.trans, path = output), "Done")
      expect_message(nc1 <- load_ncfs(output), "Done")
      is_equal_flowSet(nc.trans, nc1)
            
})


test_that("read.ncdfFlowSet: channel_alias", {
  
  data(GvHD)
  fr1 <- GvHD[[1]]
  fr2 <- GvHD[[2]]
  
  colnames(fr1)[c(3,5)] <-  c("AL1-H", "AL3-H")
  
  ## now write out into  files
  fcs1 <- tempfile()
  write.FCS(fr1, fcs1)
  fcs2 <- tempfile()
  write.FCS(fr2, fcs2)
  
  expect_message(fs <- read.ncdfFlowSet(c(fcs1,fcs2)),regexp = "Only load the following common channels")
  #strict matching by full name
  map <- data.frame(alias = c("FL1", "FL3"), channels = c("AL1-H, FL1-H", "FL3-H, AL3-H"))
  suppressMessages(fs <- read.ncdfFlowSet(c(fcs1,fcs2), channel_alias = map))
  expect_equal(colnames(fs)[c(3,5)], c("FL1", "FL3"))
  
  #partial matching
  map <- data.frame(alias = c("FL1", "FL3"), channels = c("AL1, FL1", "FL3, AL3"))
  suppressMessages(fs <- read.ncdfFlowSet(c(fcs1,fcs2), channel_alias = map))
  expect_equal(colnames(fs)[c(3,5)], c("FL1", "FL3"))
  
  #case insensitive matching
  map <- data.frame(alias = c("FL1", "FL3"), channels = c("al1, FL1", "fl3, AL3"))
  suppressMessages(fs <- read.ncdfFlowSet(c(fcs1,fcs2), channel_alias = map))
  expect_equal(colnames(fs)[c(3,5)], c("FL1", "FL3"))
  
  #ambigous partial matching
  map <- data.frame(alias = c("FL1", "FL3"), channels = c("l1, FL1", "fl3, AL3"))
  expect_error(fs <- read.ncdfFlowSet(c(fcs1,fcs2), channel_alias = map), "multiple entries")
  
  
})
