% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tune.plsda.R
\name{tune.plsda}
\alias{tune.plsda}
\title{Tuning functions for PLS-DA method}
\usage{
tune.plsda(
  X,
  Y,
  ncomp = 1,
  scale = TRUE,
  logratio = c("none", "CLR"),
  max.iter = 100,
  tol = 1e-06,
  near.zero.var = FALSE,
  multilevel = NULL,
  validation = "Mfold",
  folds = 10,
  nrepeat = 1,
  signif.threshold = 0.01,
  dist = "all",
  auc = FALSE,
  progressBar = FALSE,
  light.output = TRUE,
  BPPARAM = SerialParam(),
  seed = NULL
)
}
\arguments{
\item{X}{numeric matrix of predictors. \code{NA}s are allowed.}

\item{Y}{\code{if(method = 'spls')} numeric vector or matrix of continuous
responses (for multi-response models) \code{NA}s are allowed.}

\item{ncomp}{the number of components to include in the model.}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{logratio}{one of ('none','CLR'). Default to 'none'}

\item{max.iter}{integer, the maximum number of iterations.}

\item{tol}{Convergence stopping value.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Default value is FALSE}

\item{multilevel}{Design matrix for multilevel analysis (for repeated
measurements) that indicates the repeated measures on each individual, i.e.
the individuals ID. See Details.}

\item{validation}{character. What kind of (internal) validation to use,
matching one of \code{"Mfold"} or \code{"loo"} (short for 'leave-one-out').
Default is \code{"Mfold"}.}

\item{folds}{the folds in the Mfold cross-validation. See Details.}

\item{nrepeat}{Number of times the Cross-Validation process is repeated.}

\item{signif.threshold}{numeric between 0 and 1 indicating the significance
threshold required for improvement in error rate of the components. Default
to 0.01.}

\item{dist}{distance metric to use for \code{splsda} to estimate the
classification error rate, should be a subset of \code{"centroids.dist"},
\code{"mahalanobis.dist"} or \code{"max.dist"} (see Details).}

\item{auc}{if \code{TRUE} calculate the Area Under the Curve (AUC)}

\item{progressBar}{by default set to \code{TRUE} to output the progress bar
of the computation.}

\item{light.output}{if set to FALSE, the prediction/classification of each
sample for each of \code{test.keepX} and each comp is returned.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object indicating the type
of parallelisation. See examples in \code{?tune.spca}.}

\item{seed}{set a number here if you want the function to give reproducible outputs. 
Not recommended during exploratory analysis. Note if RNGseed is set in 'BPPARAM', this will be overwritten by 'seed'.}
}
\value{
matrix of classification error rate estimation. 
The dimensions correspond to the components in the
model and to the prediction method used, respectively.

\item{auc}{Averaged AUC values
over the \code{nrepeat}}

\item{cor.value}{only if multilevel analysis with 2 factors: correlation
between latent variables.}
}
\description{
Computes M-fold or Leave-One-Out Cross-Validation scores on a user-input
grid to determine optimal values for the parameters in \code{plsda}.
}
\details{
This tuning function should be used to tune the parameters in the
\code{plsda} function (number of components and distance metric to select).

For a PLS-DA, M-fold or LOO cross-validation is performed with stratified
subsampling where all classes are represented in each fold.

If \code{validation = "loo"}, leave-one-out cross-validation is performed.
By default \code{folds} is set to the number of unique individuals.

The function outputs the optimal number of components that achieve the best
performance based on the overall error rate or BER. The assessment is
data-driven and similar to the process detailed in (Rohart et al., 2016),
where one-sided t-tests assess whether there is a gain in performance when
adding a component to the model. Our experience has shown that in most case,
the optimal number of components is the number of categories in \code{Y} -
1, but it is worth tuning a few extra components to check (see our website
and case studies for more details).

For PLS-DA multilevel one-factor analysis, M-fold or LOO cross-validation
is performed where all repeated measurements of one sample are in the same
fold. Note that logratio transform and the multilevel analysis are performed
internally and independently on the training and test set.

For a PLS-DA multilevel two-factor analysis, the correlation between
components from the within-subject variation of X and the \code{cond} matrix
is computed on the whole data set. The reason why we cannot obtain a
cross-validation error rate as for the pls-DA one-factor analysis is
because of the difficulty to decompose and predict the within matrices
within each fold.

For a PLS two-factor analysis a PLS canonical mode is run, and the
correlation between components from the within-subject variation of X and Y
is computed on the whole data set.

If \code{validation = "Mfold"}, M-fold cross-validation is performed. How
many folds to generate is selected by specifying the number of folds in
\code{folds}.

If \code{auc = TRUE} and there are more than 2 categories in \code{Y}, the
Area Under the Curve is averaged using one-vs-all comparison. Note however
that the AUC criteria may not be particularly insightful as the prediction
threshold we use in PLS-DA differs from an AUC threshold (PLS-DA relies on
prediction distances for predictions, see \code{?predict.plsda} for more
details) and the supplemental material of the mixOmics article (Rohart et
al. 2017).

BER is appropriate in case of an unbalanced number of samples per class as
it calculates the average proportion of wrongly classified samples in each
class, weighted by the number of samples in each class. BER is less biased
towards majority classes during the performance assessment.

More details about the prediction distances in \code{?predict} and the
supplemental material of the mixOmics article (Rohart et al. 2017).

The tune.plsda() function calls older function perf() to perform this cross-validation,
for more details see the perf() help pages.
}
\examples{
## Example: analysis with PLS-DA
data(breast.tumors)

# tune components and distance
tune = tune.plsda(breast.tumors$gene.exp, as.factor(breast.tumors$sample$treatment), 
                  ncomp = 5, logratio = "none",
                  nrepeat = 10, folds = 10,
                  progressBar = TRUE,
                  seed = 20) # set for reproducibility of example only
plot(tune) # optimal distance = centroids.dist
tune$choice.ncomp # optimal component number = 3

## Example: multilevel PLS-DA
data(vac18)
design <- data.frame(sample = vac18$sample) # set the multilevel design

tune1 <- tune.plsda(vac18$genes, vac18$stimulation, 
                    ncomp = 5, multilevel = design,
                   nrepeat = 10, folds = 10,
                   seed = 20)
plot(tune1)
}
\references{
mixOmics article:

Rohart F, Gautier B, Singh A, Lê Cao K-A. mixOmics: an R package for 'omics
feature selection and multiple data integration. PLoS Comput Biol 13(11):
e1005752
}
\seealso{
\code{\link{splsda}}, \code{\link{predict.splsda}} and
http://www.mixOmics.org for more details.
}
\author{
Kim-Anh Lê Cao, Benoit Gautier, Francois Bartolo, Florian Rohart,
Al J Abadi
}
\keyword{multivariate}
\keyword{regression}
