% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plsda.R
\name{plsda}
\alias{plsda}
\title{Partial Least Squares Discriminant Analysis (PLS-DA).}
\usage{
plsda(
  X,
  Y,
  ncomp = 2,
  scale = TRUE,
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  logratio = c("none", "CLR"),
  multilevel = NULL,
  all.outputs = TRUE
)
}
\arguments{
\item{X}{numeric matrix of predictors with the rows as individual
observations. missing values (\code{NA}s) are allowed.}

\item{Y}{a factor or a class vector for the discrete outcome.}

\item{ncomp}{Positive Integer. The number of components to include in the
model. Default to 2.}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{tol}{Positive numeric used as convergence criteria/tolerance during the
iterative process. Default to \code{1e-06}.}

\item{max.iter}{Integer, the maximum number of iterations. Default to  100.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Setting this argument to FALSE (when appropriate) will speed up the
computations. Default value is FALSE.}

\item{logratio}{Character, one of ('none','CLR') specifies
the log ratio transformation to deal with compositional values that may
arise from specific normalisation in sequencing data. Default to 'none'.
See \code{?logratio.transfo} for details.}

\item{multilevel}{sample information for multilevel decomposition for
repeated measurements. A numeric matrix or data frame indicating the
repeated measures on each individual, i.e. the individuals ID. See examples
in \code{?splsda}.}

\item{all.outputs}{Logical. Computation can be faster when some specific
(and non-essential) outputs are not calculated. Default = \code{TRUE}.}
}
\value{
\code{plsda} returns an object of class \code{"plsda"}, a list that
contains the following components:

\item{X}{the centered and standardized original predictor matrix.}
\item{Y}{the centered and standardized indicator response vector or matrix.}
\item{ind.mat}{the indicator matrix.} \item{ncomp}{the number of components
included in the model.} \item{variates}{list containing the \code{X} and
\code{Y} variates.} \item{loadings}{list containing the estimated loadings
associated to each component/variate. The loading weights multiplied with the
deflated (residual) matrix gives the variate.} \item{loadings.stars}{list
containing the estimated loadings associated to each component/variate. The
loading weights are projected so that when multiplied with the original
matrix we obtain the variate.} \item{names}{list containing the names to be
used for individuals and variables.} \item{nzv}{list containing the zero- or
near-zero predictors information.} \item{tol}{the tolerance used in the
iterative algorithm, used for subsequent S3 methods} \item{max.iter}{the
maximum number of iterations, used for subsequent S3 methods}
\item{iter}{Number of iterations of the algorithm for each component}
\item{prop_expl_var}{The proportion of the variance explained by each
variate / component divided by the total variance in the \code{data} (after
removing the possible missing values) using the definition of 'redundancy'.
Note that contrary to \code{PCA}, this amount may not decrease in the
following components as the aim of the method is not to maximise the
variance, but the covariance between data sets (including the dummy matrix
representation of the outcome variable in case of the supervised
approaches).}\item{mat.c}{matrix of coefficients from the regression of X /
residual matrices X on the X-variates, to be used internally by
\code{predict}.} \item{defl.matrix}{residual matrices X for each dimension.}
}
\description{
Function to perform standard Partial Least Squares regression to classify
samples.
}
\details{
\code{plsda} function fit PLS models with \eqn{1,...,}\code{ncomp} components
to the factor or class vector \code{Y}. The appropriate indicator matrix is created.

Logratio transformation and multilevel analysis are
performed sequentially as internal pre-processing step, through
\code{\link{logratio.transfo}} and \code{\link{withinVariation}}
respectively. Logratio can only be applied if the data do not contain any 0 value (for
count data, we thus advise the normalise raw data with a 1 offset).

The type of deflation used is \code{'regression'} for discriminant algorithms.
i.e. no deflation is performed on Y.
}
\examples{
## First example
data(breast.tumors)
X <- breast.tumors$gene.exp
Y <- breast.tumors$sample$treatment

plsda.breast <- plsda(X, Y, ncomp = 2)
plotIndiv(plsda.breast, ind.names = TRUE, ellipse = TRUE, legend = TRUE)


\dontrun{
## Second example
data(liver.toxicity)
X <- liver.toxicity$gene
Y <- liver.toxicity$treatment[, 4]

plsda.liver <- plsda(X, Y, ncomp = 2)
plotIndiv(plsda.liver, ind.names = Y, ellipse = TRUE, legend =TRUE)
}
}
\references{
On PLSDA: Barker M and Rayens W (2003). Partial least squares
for discrimination. \emph{Journal of Chemometrics} \bold{17}(3), 166-173.
Perez-Enciso, M. and Tenenhaus, M. (2003). Prediction of clinical outcome
with microarray data: a partial least squares discriminant analysis (PLS-DA)
approach. \emph{Human Genetics} \bold{112}, 581-592. Nguyen, D. V. and
Rocke, D. M. (2002). Tumor classification by partial least squares using
microarray gene expression data. \emph{Bioinformatics} \bold{18}, 39-50. On
log ratio transformation: Filzmoser, P., Hron, K., Reimann, C.: Principal
component analysis for compositional data with outliers. Environmetrics
20(6), 621-632 (2009) Lê Cao K.-A., Costello ME, Lakis VA, Bartolo, F,Chua
XY, Brazeilles R, Rondeau P. MixMC: Multivariate insights into Microbial
Communities. PLoS ONE, 11(8): e0160169 (2016). On multilevel decomposition:
Westerhuis, J.A., van Velzen, E.J., Hoefsloot, H.C., Smilde, A.K.:
Multivariate paired data analysis: multilevel plsda versus oplsda.
Metabolomics 6(1), 119-128 (2010) Liquet, B., Lê Cao K.-A., Hocini, H.,
Thiebaut, R.: A novel approach for biomarker selection and the integration
of repeated measures experiments from two assays. BMC bioinformatics 13(1),
325 (2012)
}
\seealso{
\code{\link{splsda}}, \code{\link{summary}},
\code{\link{plotIndiv}}, \code{\link{plotVar}}, \code{\link{predict}},
\code{\link{perf}}, \code{\link{mint.block.plsda}},
\code{\link{block.plsda}} and http://mixOmics.org for more details.
}
\author{
Ignacio González, Kim-Anh Lê Cao, Florian Rohart, Al J Abadi
}
\keyword{multivariate}
\keyword{regression}
