% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/labelRows.R
\name{labelRows}
\alias{labelRows}
\alias{reduceTable}
\title{Annotate and Remove Report Rows}
\usage{
labelRows(
  object,
  useID = FALSE,
  minScore = 0.5,
  maxRankX = 3,
  maxRankY = 3,
  delta = 0.1,
  method = c("score", "mzrt"),
  maxRTerr = 10,
  resolveConflicts = FALSE,
  rtOrder = TRUE,
  remove = FALSE,
  balanced = TRUE,
  brackets_ignore = c("(", "[", "{")
)

reduceTable(
  object,
  useID = FALSE,
  maxRankX = 2,
  maxRankY = 2,
  minScore = 0.5,
  delta = 0.1,
  method = c("score", "mzrt"),
  maxRTerr = 10,
  rtOrder = TRUE,
  brackets_ignore = c("(", "[", "{")
)
}
\arguments{
\item{object}{Either a \code{metabCombiner} object or \code{combinedTable}}

\item{useID}{option to annotate identity-matched strings as "IDENTITY"}

\item{minScore}{numeric minimum allowable score (between 0 & 1) for
metabolomics feature pair alignments}

\item{maxRankX}{integer maximum allowable rank for X dataset features.}

\item{maxRankY}{integer maximum allowable rank for Y dataset features.}

\item{delta}{numeric score or mz/rt distances used to define subgroups. If
method = "score", a value (between 0 & 1) score difference
between a pair of conflicting FPAs. If method = "mzrt", a length
4 numeric: (m/z, rt, m/z, rt) tolerances, the first pair for X
dataset features and the second pair for Y dataset features.}

\item{method}{Conflict detection method. If equal to "score" (default),
assigns a conflict subgroup if score of lower-ranking FPA is
within some tolerance of higher-ranking FPA. If set to "mzrt",
assigns a conflicting subgroup if within a small m/z & rt
distance of the top-ranked FPA.}

\item{maxRTerr}{numeric maximum allowable error between model-projected
retention time (rtProj) and observed retention time (rty)}

\item{resolveConflicts}{logical option to computationally resolve conflicting
rows to a final set of 1-1 feature pair alignments}

\item{rtOrder}{logical. If resolveConflicts set to TRUE, then this imposes
retention order consistency on rows deemed "RESOLVED" within subgroups.}

\item{remove}{Logical. Option to keep or discard rows deemed removable.}

\item{balanced}{Logical. Optional processing of "balanced" groups, defined
as groups with an equal number of features from input
datasets where all features have a 1-1 match.}

\item{brackets_ignore}{character. If useID = TRUE, bracketed identity strings
of the types in this argument will be ignored}
}
\value{
updated \code{combinedTable} or \code{metabCombiner} object. The
table will have three new columns:

\item{labels}{characterization of feature alignments as described}
\item{subgroup}{conflicting subgroup number of feature alignments}
\item{alt}{alternate subgroup for rows in multiple feature pair conflicts}
}
\description{
This is a method for annotating removable, conflicting, and identity-matched
feature pair alignment (FPA) rows in the \code{combinedTable} report. Simple
thresholds for score, rank, retention time error and delta score can
computationally reduce the set of possible FPAs to the most likely feature
matches. FPAs falling within some small delta score or mz/rt of the
top-ranked pair are organized into subgroups to facilitate inspection.
Automated reduction to 1-1 pairs is also possible with this function.

\code{reduceTable} behaves identically to labelRows, but with a focus on
automated table reduction. Rank threshold defaults in \code{reduceTable} are
also stricter than in \code{labelRows}.
}
\details{
\code{metabCombiner} initially reports all possible feature pairings in the
rows of the \code{combinedTable} report. Most of these are misalignments that
require removal. This function is used to automate this reduction
process by labeling rows as removable or conflicting, based on certain
conditions, and is performed after computing similarity scores.

A label may take on one of four values:

 a) "": No determination made
 b) "IDENTITY": an alignment with matching identity "idx & idy" strings
 c) "REMOVE": a row determined to be a misalignment
 d) "CONFLICT": competing alignments for one or multiple shared features

The labeling rules are as follows:

1) Groups determined to be 'balanced': label rows with rankX > 1 & rankY > 1
   "REMOVE" irrespective of \code{delta} criteria
2) Rows with a score < \code{minScore}: label "REMOVE"
3) Rows with rankX > \code{maxRankX} and/or rankY > \code{maxRankY}:
   label "REMOVE"
4) Conflicting subgroup assignment as determined  by \code{method} &
   \code{delta} arguments. Conflicting alignments following outside
   \code{delta} thresholds: labeled "REMOVE". Otherwise, they are assigned
   a "CONFLICT" label and subgroup number.
5) If \code{useID} argument set to TRUE, rows with matching idx & idy strings
   are labeled "IDENTITY". These rows are not changed to "REMOVE" or
   "CONFLICT" irrespective of subsequent criteria.
}
\examples{

#required steps prior to function use
data(plasma30)
data(plasma20)
p30 <- metabData(plasma30, samples = "CHEAR")
p20 <- metabData(plasma20, samples = "Red", rtmax = 17.25)
p.comb <- metabCombiner(xdata = p30, ydata = p20, binGap = 0.0075)
p.comb <- selectAnchors(p.comb, tolmz = 0.003, tolQ = 0.3, windy = 0.02)
p.comb <- fit_gam(p.comb, k = 20, iterFilter = 1)
p.comb <- calcScores(p.comb, A = 90, B = 14, C = 0.5)

##applies labels, but maintains all rows
p.comb <- labelRows(p.comb, maxRankX = 2, maxRankY = 2, maxRTerr = 0.5,
                    delta = 0.1, resolveConflicts = FALSE, remove = FALSE)

##automatically resolve conflicts and filter to 1-1 feature pairs
p.comb.2 <- labelRows(p.comb, resolveConflicts = FALSE, remove = FALSE)

#this is identical to the previous command
p.comb.2 <- reduceTable(p.comb)

p.comb <- labelRows(p.comb, method = "mzrt", delta = c(0.005, 0.5, 0.005,0.3))

##this function may be applied to combinedTable inputs as well
cTable <- cbind.data.frame(combinedTable(p.comb), featData(p.comb))

lTable <- labelRows(cTable, maxRankX = 3, maxRankY = 2, minScore = 0.5,
         method = "score", maxRTerr = 0.5, delta = 0.2)

}
