% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CohenD_IndividualGenes.R
\name{CohenD_IndividualGenes}
\alias{CohenD_IndividualGenes}
\title{Cohen's d Heatmap Function}
\usage{
CohenD_IndividualGenes(
  data,
  metadata,
  genes = NULL,
  condition_var,
  class,
  group_var = NULL,
  title = NULL,
  titlesize = 16,
  params = list()
)
}
\arguments{
\item{data}{A data frame or matrix containing gene expression data, with
genes as rows and samples as columns.}

\item{metadata}{A data frame containing sample metadata. The first column
should contain sample identifiers that match the column names of
\code{data}.}

\item{genes}{A character vector specifying which genes to include. If
\code{NULL} (default), all genes in \code{data} are used. A warning is
issued if more than 30 genes are selected.}

\item{condition_var}{A character string specifying the column name in
\code{metadata} representing the condition of interest. (Mandatory; no
default.)}

\item{class}{A character string or vector specifying the positive class label
for the condition. (Mandatory; no default.)}

\item{group_var}{An optional character string specifying the column name in
\code{metadata} used for grouping samples. If not provided (\code{NULL}),
all samples are treated as a single group.}

\item{title}{An optional character string specifying a custom title for the
heatmap. If not provided, a default title is generated.}

\item{titlesize}{A numeric value specifying the size of the title. Default is
\code{14}.}

\item{params}{A list of additional parameters for customizing the heatmap.
Possible elements include:
\describe{
\item{\code{cluster_rows}}{Logical; if \code{TRUE} (default), rows are
clustered.}
\item{\code{cluster_columns}}{Logical; if \code{TRUE} (default), columns
are clustered.}
\item{\code{colors}}{A vector of length 2 of colors to be used for the
minimum and maximum values of the color scale.
Defaults to \code{c("#FFFFFF", "#21975C")}, but note that the default
mapping for Cohen's d is set to a divergent scale.}
\item{\code{limits}}{A numeric vector of length 2 specifying the minimum
and maximum values for the color scale.
If not provided, defaults to \code{c(-2, 2)}.}
\item{\code{name}}{A character string for the legend title of the color
scale. Default is \code{"Cohen's d"}.}
\item{\code{row_names_gp}}{Optional graphical parameters for row names
(passed to \pkg{ComplexHeatmap}).}
\item{\code{column_names_gp}}{Optional graphical parameters for column
names (passed to \pkg{ComplexHeatmap}).}
}}
}
\value{
Invisibly returns a list containing:
\describe{
\item{\code{plot}}{The \pkg{ComplexHeatmap} object of the Cohen's d heatmap.}
\item{\code{data}}{A data frame with the calculated Cohen's d values for
each gene and group.}
}
}
\description{
This function computes Cohen's d for each gene based on gene expression data
and sample metadata. For each gene, it compares the expression values between
samples where \code{condition_var} equals \code{class} (the positive class)
versus the remaining samples. The resulting effect sizes are then visualized
as a heatmap.
}
\details{
This function computes Cohen's d for each gene by comparing the
expression values between samples with \code{condition_var == class} and
those that do not. The effect sizes are then visualized as a heatmap using
\pkg{ComplexHeatmap}. When \code{group_var} is not provided, all samples are
treated as a single group.
}
\examples{
# Simulate gene expression data (genes as rows, samples as columns)
set.seed(101)
expr <- matrix(abs(rnorm(40)), nrow = 4, ncol = 10)  # 4 genes, 10 samples,
# positive values
rownames(expr) <- paste0("Gene", 1:4)
colnames(expr) <- paste0("Sample", 1:10)

# Simulate sample metadata with a binary condition and a group variable
metadata <- data.frame(
  Sample = colnames(expr),
  Condition = rep(c("Case", "Control"), each = 5),
  Group = rep(c("A", "B"), times = 5),
  stringsAsFactors = FALSE
)

# 1. Cohen's d heatmap for all genes across groups
CohenD_IndividualGenes(
  data = expr,
  metadata = metadata,
  genes = rownames(expr),
  condition_var = "Condition",
  class = "Case",
  group_var = "Group",
  title = "Cohen's d Heatmap (Case vs Control)",
  params = list(limits = c(0, 2))
)

# 2. Cohen's d barplot (single group across all samples)
CohenD_IndividualGenes(
  data = expr,
  metadata = metadata,
  genes = rownames(expr),
  condition_var = "Condition",
  class = "Case",
  group_var = NULL,
  title = "Cohen's d Barplot (All Samples)",
  params = list(colors = c("#CCCCCC"))
)


}
