#' Plot a Venn Diagram from Genomic or Set Overlap Results
#'
#' This function creates a Venn diagram using the `eulerr` package to visualize
#' intersections across multiple sets. Supports both
#' `GenomicOverlapsResult` and `SetOverlapsResult` objects.
#'
#' @param overlap_object A `GenomicOverlapsResult` or `SetOverlapsResult` object
#'   returned by \code{\link{computeOverlaps}}.
#' @param fills Controls the fill appearance of the diagram. Can be:
#'   \itemize{
#'     \item \strong{logical}: `TRUE` (default) shows fills, `FALSE` hides them
#'     \item \strong{character vector}: Colors for the fills.
#'     Default colors are: c("#2B70AB", "#FFB027", "#3EA742", "#CD3301", "#9370DB", "#008B8B", "#D87093")
#'     \item \strong{list}: Fine control with graphical parameters including
#'       `fill` (colors), `alpha` (transparency 0-1)
#'   }
#' @param edges Controls the edge/border appearance. Can be:
#'   \itemize{
#'     \item \strong{logical}: `TRUE` (default) shows edges, `FALSE` hides them
#'     \item \strong{character vector}: Colors for the edges
#'     \item \strong{list}: Fine control with `col` (colors), `alpha` (transparency 0-1),
#'       `lty` (line type), `lwd` (line width), `lex` (line expansion)
#'   }
#' @param labels Controls set labels. Can be:
#'   \itemize{
#'     \item \strong{logical}: `TRUE` shows default labels, `FALSE` hides them
#'     \item \strong{character vector}: Custom text for labels
#'     \item \strong{list}: Fine control with `col` (text color), `fontsize`,
#'       `font` (1=plain, 2=bold, 3=italic, 4=bold italic), `fontfamily`,
#'       `cex` (character expansion), `alpha` (transparency 0-1)
#'   }
#' @param quantities Controls intersection quantities display. Can be:
#'   \itemize{
#'     \item \strong{logical}: `TRUE` shows counts, `FALSE` hides them
#'     \item \strong{character vector}: Custom text labels
#'     \item \strong{list}: Fine control with `type` (c("counts", "percent")),
#'       `col` (text color), `fontsize`, `font`, `fontfamily`, `cex`, `alpha`
#'   }
#' @param legend Controls the legend. Can be:
#'   \itemize{
#'     \item \strong{logical}: `FALSE` to disable
#'     \item \strong{character}: Position ("right", "top", "bottom", "left")
#'     \item \strong{list}: Fine control with `side` (position), `labels`
#'       (custom labels), `col`, `fontsize`, `cex`, `fontfamily`
#'   }
#' @param main Title for the plot. Can be character, expression, or list with
#'   `label` (text), `col`, `fontsize`, `font`, `fontfamily`
#' @param ... Additional arguments passed to \code{\link[eulerr]{plot.euler}}.
#'
#' @return A Venn diagram plot generated by \code{eulerr}.
#' @export
#'
#' @examples
#' # Example with gene sets
#' data(gene_list)
#' res_sets <- computeOverlaps(gene_list)
#'
#' # Basic plot
#' plotVenn(res_sets)
#'
#' # Customize fills with transparency and custom colors
#' plotVenn(res_sets,
#'          fills = list(fill = c("#FF6B6B", "#4ECDC4", "#45B7D1"),
#'                       alpha = 0.6))
#'
#' # Customize edges
#' plotVenn(res_sets,
#'          edges = list(col = "darkgray", lwd = 2, lty = 2))
#'
#' # Customize labels
#' plotVenn(res_sets,
#'          labels = list(col = "white", font = 2, fontsize = 14))
#'
#' # Show both counts and percentages
#' plotVenn(res_sets,
#'          quantities = list(type = c("counts", "percent"),
#'                           col = "black", fontsize = 10))
#'
#' # Add a title
#' plotVenn(res_sets,
#'          main = list(label = "Gene Set Overlaps",
#'                      col = "navy", fontsize = 16, font = 2))
#'
#' # Transparent fills with colored borders only
#' plotVenn(res_sets,
#'          fills = "transparent",
#'          edges = list(col = c("red", "blue", "green"), lwd = 3))
#'
#' # Custom legend
#' plotVenn(res_sets,
#'          legend = list(side = "bottom",
#'                       labels = c("Treatment A", "Treatment B", "Control"),
#'                       fontsize = 12))
plotVenn <- function(overlap_object,
                     fills = TRUE,
                     edges = TRUE,
                     labels = FALSE,
                     quantities = list(type = "counts"),
                     legend = "right",
                     main = NULL,
                     ...) {

    # Check input object class
    if (!inherits(overlap_object, "GenomicOverlapResult") &&
        !inherits(overlap_object, "SetOverlapResult")) {
        stop("Input must be a GenomicOverlapResult or SetOverlapResult.")
    }

    # Extract matrix
    overlap_matrix <- overlap_object[["overlap_matrix"]]

    # Compute Euler fit
    fit <- eulerr::euler(overlap_matrix, shape = "ellipse")

    # Process fills parameter
    if (is.logical(fills) && fills == TRUE) {
        # Default colors if fills = TRUE
        default_colors <- c("#2B70AB", "#FFB027", "#3EA742", "#CD3301",
                            "#9370DB", "#008B8B", "#D87093")
        fills <- default_colors
    } else if (is.character(fills) && length(fills) == 1 && fills == "transparent") {
        # Handle "transparent" as special case
        fills <- "transparent"
    } else if (is.character(fills)) {
        # User provided colors as vector
        fills <- fills
    }
    # If fills is already a list or FALSE, pass through as-is

    # Process edges parameter
    if (is.logical(edges) && edges == TRUE) {
        # Default edges settings
        edges <- list(alpha = 0)  # Default to invisible edges when fills are used
    }
    # If edges is already a list, character vector, or FALSE, pass through as-is

    # Process legend parameter
    if (is.character(legend)) {
        # Convert position string to list format
        legend <- list(side = legend)
    } else if (is.logical(legend) && legend == FALSE) {
        legend <- FALSE
    }
    # If legend is already a list, pass through as-is

    # Validate quantities type when it's a list
    if (is.list(quantities) && !is.null(quantities$type)) {
        ok <- c("counts", "percent")
        qt <- as.character(quantities$type)
        if (!all(qt %in% ok) || length(qt) > 2) {
            stop("`quantities$type` must be a vector of 'counts' and/or ",
                 "'percent' (max length 2).")
        }
    }

    # Plot the diagram
    venn <- plot(
        fit,
        fills = fills,
        edges = edges,
        labels = labels,
        quantities = quantities,
        legend = legend,
        main = main,
        ...
    )

    return(venn)
}
