
#' mkdSetGlobalOpts
#' @description internal function for appending to the report the initial
#' options setup
#'
#' @param object an easyreporting class object
#' @param optionList a list of options
#'
#' @return an easyreporting class object
#' @keywords internal
#'
setMethod(f="mkdSetGlobalOpts", signature="easyreporting",
        definition=function(object, optionList=list())
{
    if(!is.null(optionList)) object@optionList <- optionList
    options <- paste0("```{r global_options, include=FALSE}\n",
                        "knitr::opts_chunk$set(",
                        "eval=", object@optionList$eval,
                        ", echo=", object@optionList$echo,
                        ", warning=", object@optionList$warning,
                        ", message=", object@optionList$showMessages,
                        ", include=", object@optionList$include,
                        ", cache=", object@optionList$cache,
                        ", collapse=", object@optionList$collapse,
                        ", purl=", object@optionList$purl,
                        ", error=", object@optionList$error,
                        ", message=", object@optionList$message,
                        ", highlight=", object@optionList$highlight,
                        ", prompt=", object@optionList$prompt,
                        ", strip.white=", object@optionList$strip.white,
                        ", tidy=", object@optionList$tidy,
                        ")\n```\n")
    
    base::write(options, file=object@filenamePath,
    append=TRUE, sep="\n")
    return(object)
})


#' setOptionsList
#' @description set an optionList to the class
#' 
#' @param object an easyreporting class object
#' @param cacheFlag boolean for caching chunk data (default TRUE)
#' @param evalFlag boolean for evaluating the code chunk in the compiled version
#' (default TRUE)
#' @param echoFlag boolean for showing the code chunk (default TRUE)
#' @param warningFlag boolean for showing the chunk warnings (default FALSE)
#' @param showMessages boolean for showing the chunk warnings in compiled
#' version (default FALSE)
#' @param includeFlag boolean for including the code chunk in the compiled
#' version (default TRUE)
#' @param collapseFlag boolean for collapsing the code chunk in the compiled
#' version (default FALSE),
#' @param purlFlag boolean for extracting the code chunk as R code in a separate
#' R file (default TRUE),
#' @param errorFlag boolean for including the error generated by the code chunk 
#' in the compiled version (default TRUE),
#' @param messageFlag boolean for including the code chunk messages in the compiled
#' version (default TRUE),
#' @param highlightFlag boolean for highlinghtinh the code chunk in the compiled
#' version (default TRUE),
#' @param promptFlag boolean for including a ">" for the code chunk in the compiled
#' version (default FALSE),
#' @param stripWhiteFlag boolean for removing the white spaces at beginning/end 
#' of the code chunk in the compiled version (default TRUE),
#' @param tidyFlag boolean for creating a tidy code chunk in the compiled
#' version (default FALSE).
#'
#' @return none
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report",
#'                         title="example_report", author=c("It's me"))
#'
#' ## setting default option
#' setOptionsList(rd)
#'
#' ## modifying only some options
#' rd <- setOptionsList(rd, warningFlag=TRUE, 
#'         showMessages=TRUE, includeFlag=TRUE)
#' }
setMethod(f="setOptionsList", signature="easyreporting", 
        definition=function(
            object, 
            cacheFlag=TRUE,
            evalFlag=TRUE,
            echoFlag=TRUE,
            warningFlag=FALSE,
            showMessages=FALSE,
            includeFlag=TRUE,
            collapseFlag=FALSE,
            purlFlag=TRUE,
            errorFlag=TRUE,
            messageFlag=TRUE,
            highlightFlag=TRUE,
            promptFlag=FALSE,
            stripWhiteFlag=TRUE,
            tidyFlag=FALSE
            )
{
    object@optionList <- list(
        cache=cacheFlag,
        eval=evalFlag,
        echo=echoFlag,
        warning=warningFlag,
        showMessages=showMessages,
        include=includeFlag,
        collapse=collapseFlag,
        purl=purlFlag,
        error=errorFlag,
        message=messageFlag,
        highlight=highlightFlag,
        prompt=promptFlag,
        strip.white=stripWhiteFlag,
        tidy=tidyFlag
    )
    return(object)
})

#' getOptionsList
#'
#' @description returns the optionList from the easyreporting class
#'
#' @param object an easyreporting class object
#'
#' @return a list of options
#' @export
#'
#' @examples
#' \dontrun{
#' rd <- easyreporting(filenamePath="./project_report", title="example_report",
#'                         author=c("It's me"))
#' optList <- getOptionsList(rd)
#' }
setMethod(f="getOptionsList", signature="easyreporting", 
        definition=function(object)
        {
            return(object@optionList)
        }
)


#' makeOptionsList
#' @description makes an list of rmarkdown options
#'
#' @param cacheFlag boolean for caching chunk data (default TRUE)
#' @param evalFlag boolean for evaluating the code chunk in the compiled version
#' (default TRUE)
#' @param echoFlag boolean for showing the code chunk (default TRUE)
#' @param warningFlag boolean for showing the chunk warnings (default FALSE)
#' @param showMessages boolean for showing the chunk warnings in compiled
#' version (default FALSE)
#' @param includeFlag boolean for including the code chunk in the compiled
#' version (default TRUE)
#' @param collapseFlag boolean for collapsing the code chunk in the compiled
#' version (default FALSE),
#' @param purlFlag boolean for extracting the code chunk as R code in a separate
#' R file (default TRUE),
#' @param errorFlag boolean for including the error generated by the code chunk 
#' in the compiled version (default TRUE),
#' @param messageFlag boolean for including the code chunk messages in the compiled
#' version (default TRUE),
#' @param highlightFlag boolean for highlinghtinh the code chunk in the compiled
#' version (default TRUE),
#' @param promptFlag boolean for including a ">" for the code chunk in the compiled
#' version (default FALSE),
#' @param stripWhiteFlag boolean for removing the white spaces at beginning/end 
#' of the code chunk in the compiled version (default TRUE),
#' @param tidyFlag boolean for creating a tidy code chunk in the compiled
#' version (default FALSE).
#'
#' @return list of rmarkdown options
#' @export
#'
#' @examples
#' optList <- makeOptionsList()
makeOptionsList <- function(cacheFlag=TRUE,
                            evalFlag=TRUE,
                            echoFlag=TRUE,
                            warningFlag=FALSE,
                            showMessages=FALSE,
                            includeFlag=TRUE,
                            collapseFlag=FALSE,
                            purlFlag=TRUE,
                            errorFlag=TRUE,
                            messageFlag=TRUE,
                            highlightFlag=TRUE,
                            promptFlag=FALSE,
                            stripWhiteFlag=TRUE,
                            tidyFlag=FALSE)
{
    return( list(
        cache=cacheFlag,
        eval=evalFlag,
        echo=echoFlag,
        warning=warningFlag,
        showMessages=showMessages,
        include=includeFlag,
        collapse=collapseFlag,
        purl=purlFlag,
        error=errorFlag,
        message=messageFlag,
        highlight=highlightFlag,
        prompt=promptFlag,
        strip.white=stripWhiteFlag,
        tidy=tidyFlag
    ))
}
