% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict_immune_response.R
\name{predict_immune_response}
\alias{predict_immune_response}
\title{Compute predicted immune response}
\usage{
predict_immune_response(
  pathways = NULL,
  immunecells = NULL,
  tfs = NULL,
  lrpairs = NULL,
  ccpairs = NULL,
  cancer_type,
  verbose = TRUE
)
}
\arguments{
\item{pathways}{numeric matrix with pathways activity
(rows = samples; columns = pathways).}

\item{immunecells}{numeric matrix with immune cell quantification
(rows = samples; columns = cell types).}

\item{tfs}{numeric matrix with transcription factors activity
(rows = samples; columns = transcription factors).}

\item{lrpairs}{numeric matrix with ligand-receptor weights
(rows = samples; columns = ligand-receptor pairs).}

\item{ccpairs}{numeric matrix with cell-cell scores
(rows = samples; columns = cell-cell pairs).}

\item{cancer_type}{character string indicating which cancer-specific
model should be used to compute the predictions. This should be available
from the cancer-specific models. The following cancer types have a
corresponding model available: "BLCA", "BRCA", "CESC", "CRC", "GBM",
"HNSC", "KIRC", "KIRP", "LIHC", "LUAD", "LUSC", "NSCLC", "OV", "PAAD",
"PRAD", "SKCM", "STAD", "THCA" and "UCEC".}

\item{verbose}{logical flag indicating whether to display messages
about the process.}
}
\value{
A list containing the predictions for each quantitative descriptor
and for each task.
Given that the model training was repeated 100 times with randomized-cross
validation, a set of 100 predictions is returned.
}
\description{
Calculates predictions of patients' immune response
using the quantitative descriptors data as input
features and the optimized model parameters derived
from the trained models. These models are available from
easierData package through \code{easierData::get_opt_models()}.
}
\examples{
# using a SummarizedExperiment object
library(SummarizedExperiment)
# Using example exemplary dataset (Mariathasan et al., Nature, 2018)
# from easierData. Original processed data is available from
# IMvigor210CoreBiologies package.
library("easierData")

dataset_mariathasan <- easierData::get_Mariathasan2018_PDL1_treatment()
RNA_tpm <- assays(dataset_mariathasan)[["tpm"]]
cancer_type <- metadata(dataset_mariathasan)[["cancertype"]]

# Select a subset of patients to reduce vignette building time.
pat_subset <- c(
  "SAM76a431ba6ce1", "SAMd3bd67996035", "SAMd3601288319e",
  "SAMba1a34b5a060", "SAM18a4dabbc557"
)
RNA_tpm <- RNA_tpm[, colnames(RNA_tpm) \%in\% pat_subset]

# Computation of TF activity (Garcia-Alonso et al., Genome Res, 2019)
tf_activity <- compute_TF_activity(
  RNA_tpm = RNA_tpm
)

# Predict patients' immune response
predictions_immune_response <- predict_immune_response(
  tfs = tf_activity,
  cancer_type = cancer_type
)

\donttest{

RNA_counts <- assays(dataset_mariathasan)[["counts"]]
RNA_counts <- RNA_counts[, colnames(RNA_counts) \%in\% pat_subset]

# Computation of cell fractions (Finotello et al., Genome Med, 2019)
cell_fractions <- compute_cell_fractions(RNA_tpm = RNA_tpm)

# Computation of pathway scores (Holland et al., BBAGRM, 2019;
# Schubert et al., Nat Commun, 2018)
pathway_activity <- compute_pathway_activity(
  RNA_counts = RNA_counts,
  remove_sig_genes_immune_response = TRUE
)

# Computation of ligand-receptor pair weights
lrpair_weights <- compute_LR_pairs(
  RNA_tpm = RNA_tpm,
  cancer_type = "pancan"
)

# Computation of cell-cell interaction scores
ccpair_scores <- compute_CC_pairs(
  lrpairs = lrpair_weights,
  cancer_type = "pancan"
)

# Predict patients' immune response
predictions_immune_response <- predict_immune_response(
  pathways = pathway_activity,
  immunecells = cell_fractions,
  tfs = tf_activity,
  lrpairs = lrpair_weights,
  ccpairs = ccpair_scores,
  cancer_type = cancer_type
)
}
}
