\name{interpretSpheres}
\alias{interpretSpheres}

\title{Interactive interpretation of hyperspheres}
\description{Launches a Shiny app to assist interpretation of hyperspheres.}

\usage{
interpretSpheres(x, markers=NULL, labels=NULL, select=NULL, metrics=NULL, 
    num.per.row=6, plot.height=100, xlim=NULL, p=0.01, 
    red.coords=NULL, red.highlight=NULL, red.plot.height=500, 
    add.plot=NULL, add.plot.height=500, run=TRUE, ...)   
}

\arguments{
\item{x}{A CyData object generated by \code{\link{countCells}}, containing counts and coordinates for each hypersphere.}
\item{markers}{A character vector indicating the markers to use, and the order they should be plotted in. 
If \code{NULL}, all markers are used in the order corresponding to the columns of \code{intensities(x)}.}
\item{labels}{A character vector containing existing labels for the hyperspheres.
This should be of length equal to the number of rows in \code{x}.}
\item{select}{A logical or integer vector indicating which rows of \code{x} should be inspected.
Defaults to all rows.}
\item{metrics}{A dataframe containing metrics to be reported for each hypersphere, with number of rows equal to \code{x}.}
\item{num.per.row}{An integer scalar specifying the number of plots per row.}
\item{plot.height}{An integer scalar specifying the height of each plot in pixels.}
\item{xlim}{A numeric vector of length two specifying the x-axis limits for all plots.
Otherwise, \code{\link{intensityRanges}} is used to define limits for each marker.}
\item{p}{A numeric scalar to be passed to \code{\link{intensityRanges}}.}
\item{red.coords}{A numeric matrix with two columns and number of rows equal to \code{nrow(x)}, containing a reduced-dimension representation of hypersphere coordinates.
The first and second columns should contain the x- and y-coordinates, respectively.}
\item{red.highlight}{A logical or integer vector specifying which rows of \code{x} should be highlighed on the reduced dimensionality plot.}
\item{red.plot.height}{An integer scalar specifying the height of the reduced-dimension plot.}
\item{add.plot}{A function taking two arguments (see below) to create additional plots in the app.}
\item{add.plot.height}{An integer scalar specifying the height of the additional plots.}
\item{run}{A logical scalar specifying whether the Shiny app should be run.}
\item{...}{Additional arguments to be passed to \code{\link{density}}.}
}

\details{
This function creates a Shiny app in which density plots are constructed for intensities across all cells, one for each marker.
For a given hypersphere, the median intensity is plotted as a red circle on top of the density plot for each marker.
This allows users to quickly determine the biological meaning of each hypersphere, based on its median marker expression (and other statistics in \code{metrics}).

For each marker, the area under the curve is highlighted using the \code{\link{viridis}} colour scheme.
This is based on whether the median is relatively high (yellow) or low (purple) compared to all of the cells.
An interval around the median is also displayed, representing the range of intensities across a given percentage (default 95\%) of cells in the hypersphere.
This provides more information about the spread of intensities within each hypersphere.

Each hypersphere can be labelled with some meaningful term, e.g., the cell type that corresponds to the suite of expressed markers.
For each hypersphere, the closest hyperspheres that have already been labelled are shown, along with the Euclidean distances to their locations.
This is designed to assist with the labelling process by identifying pre-labelled hyperspheres in the neighbouring space.

Finally, the labels can be saved to R using the \dQuote{Save to R} button.
This stops the app and returns a character vector of labels in the R session.
Existing labels can also be re-used by supplying them to \code{labels}, to allow users to label parts of the data set at a time.
}

\section{Navigation}{
Users can navigate through the data set using the \dQuote{Previous} or \dQuote{Next} buttons.
This moves across hyperspheres specified by \code{select}, i.e., pressing \dQuote{Next} will jump to the next hypersphere in \code{select}.
By default, \code{select=NULL} which means that the app will progress through all hyperspheres in \code{x}.
It is often worth setting \code{select}, e.g., to non-redundant significant hyperspheres in order to reduce the number of elements that need to be inspected.

Users can also jump to a particular row/hypersphere by providing an integer scalar in the \dQuote{Go to sphere} field.
This specifies the row index for the hypersphere of interest, and works for either selected or unselected hyperspheres.
However, pressing \dQuote{Previous} or \dQuote{Next} will jump to the nearest index of the selected hyperspheres. 
The navigation history at any given time is shown in the side bar.

A reduced-dimensionality plot is also constructed using specified coordinates in two-dimensional space for each hypersphere.
The current hypersphere is marked on this plot with a red dot.
Previously visited and labelled hyperspheres are marked in black.
Users can also highlight particular hyperspheres in orange with \code{red.highlight}, e.g., if they are significantly differential or not.
}

\section{Putting in additional plots}{
Users can define \code{add.plot} as a function taking two arguments:
\enumerate{
\item An integer scalar, specifying the row index of the second argument.
This corresponds to the hypersphere currently being inspected.
\item A CyData object, which is set internally to the value of \code{x} used in \code{interpretSpheres}.
This should contain information about all hyperspheres.
}
\code{add.plot} should generate a plot on the current graphics device.
This is usually done in a hypersphere-specific manner, where the first argument is used to extract the relevant information from the second argument.
For example, the abundances of all samples can be visualized directly for each hypersphere in the app.
}

\section{Inspecting label propagation}{
If \code{red.dim} is supplied and at least one cell is labelled, the \dQuote{Update labels} button can be used to propagate the labels to surrounding cells.
Specifically, for each unlabelled cell, the closest labelled cell is identified and its label is assigned to the unlabelled cell.
A plot is then created showing the distribution of cells for each label in the low-dimensional space.

This functionality is useful for determining how the labels would be automatically assigned by \code{\link{labelSpheres}}.
If many distinct clusters have the same label, it suggests that more manual labelling is required to distinguish clusters.
Note that the automatically assigned labels are \emph{not} recorded, they are only used here for visualization purposes.
}

\value{
If \code{run=FALSE}, a Shiny app is returned that can be run with \code{\link{runApp}}.
This passes control to a browser window in which labels can be entered for each hypersphere.
Upon stopping the app, a character vector of length equal to the number of rows in \code{x} is returned.

If \code{run=TRUE}, a Shiny app is opened directly in a browser window.
This returns a character vector upon stopping, as previously described.
}

\author{
    Aaron Lun
}

\seealso{
    \code{\link{density}},
    \code{\link{intensityRanges}},
    \code{\link{runApp}}
}

\examples{
# Mocking up some data.
example(prepareCellData, echo=FALSE)
cnt <- countCells(cd, filter=1)

# Constructing the app
app <- interpretSpheres(cnt, run=FALSE)

\dontrun{# Running the app from the object.
labels <- shiny::runApp(app)

#Or directly running the app from the function.
labels <- interpretSpheres(cnt)
}

# Doing it with metrics and coordinates.
N <- nrow(cnt)
metrics <- data.frame(logFC=rnorm(N), PValue=runif(N))
coords <- matrix(rnorm(N*2), ncol=2)
app <- interpretSpheres(cnt, red.coord=coords, metrics=metrics, run=FALSE)

# Doing it with an extra plot.
app <- interpretSpheres(cnt, run=FALSE, add.plot=function(i, x) {
    barplot(assay(x)[i,]/x$totals*100, ylab="Percentage of cells")
})

}

