#' @param x A character vector
#'
#' @noRd
#' @keywords internal
indent_text <- function(x) {
  paste0("  ", gsub("\n", "\n  ", x, fixed = TRUE))
}

#' @param x A character vector`
#'
#' @noRd
#' @keywords internal
prepare_tests <- function(x) {
  gsub("^\\s*(.*?)\\s*$", "\\1", x)
}

#' @param x A character vector
#'
#' @noRd
#' @keywords internal
auto_quote <- function(x) {
  needs_quotes <- !grepl("^('|\").*\\1$", x) & !make.names(x) == x
  x[needs_quotes] <- encodeString(x[needs_quotes], quote = "\"")
  x
}

#' @param blocks A list of roxygen2 blocks
#'
#' @noRd
#' @keywords internal
collect_annotate_rdname <- function(blocks) {
  rdnameblocks <- list()
  otherblocks <- list()

  for (block in blocks) {
    block_rdname <- roxygen2::block_get_tags(block, "rdname")

    if (length(block_rdname) == 0L) {
      otherblocks[[length(otherblocks) + 1L]] <- block
      next
    }

    if (length(block_rdname) >= 2L) {
      stop("Did not expect two @rdname's in a block")
    }

    # length(block_rdname) == 1:
    block_rdname <- block_rdname[[1L]]

    if (is.null(rdnameblocks[[ block_rdname$val  ]])) {
      rdnameblocks[[ block_rdname$val  ]] <- block
    } else {
      rdnameblocks[[ block_rdname$val  ]]$tags <- c(
        rdnameblocks[[ block_rdname$val  ]]$tags,
        block$tags
      )
    }
  }

  c(rdnameblocks, otherblocks)
}

#' Convert a roxygen2 block to a testthat file
#'
#' This is an internal function that converts a roxygen2 block to a testthat
#' file. It first checks if the block has any 'longtests' tags. If not, it
#' returns NULL. Then, it extracts the 'longtests' and 'examples' tags from the
#' block and assigns them to the 'longtests' and 'examples' fields of the
#' testthat file, respectively. It also assigns the basename of the block's file
#' to the 'filename' field, and the alias of the block's object to the
#' 'functionname' field. If the block's line is not NULL, it appends it to the
#' 'functionname' field.
#'
#' @param block A roxygen2 block.
#'
#' @return A list representing a testthat file, or NULL if the block has no
#'   'longtests' tags.
#' @keywords internal
block_to_testthat <- function(block) {
  testthat_file <- list()
  longtests <- roxygen2::block_get_tags(block, "longtests")
  if (length(longtests) == 0L) { return(NULL) }
  testthat_file$longtests <- longtests
  testthat_file$examples <- roxygen2::block_get_tags(block, "examples")
  testthat_file$filename <- basename(block$file)
  testthat_file$functionname <- ifelse(
    !is.null(block$object) && !is.null(block$object$alias),
    paste0('Function ', auto_quote(block$object$alias), '()'),
    "[unknown alias]"
  )

  if (!is.null(block$line)) {
    testthat_file$functionname <- paste0(
      testthat_file$functionname, ' @ L', block$line
    )
  }

  testthat_file
}

#' Process test files for 'longtests'
#'
#' This is an internal function that processes a list of test files for
#' 'longtests'. It loops over each test file and extracts the 'longtests' and
#' 'functionname' fields. It then prepares the tests, optionally indents the
#' code, and formats the tests into a string with an optional 'test_that'
#' boilerplate. The function also constructs a header for the content, including
#' a 'context' line if specified. The resulting content strings are returned in
#' a named list, with the names being the paths of the test files.
#'
#' @param testfiles A named list of test files, where the names are the paths of
#'   the files.
#' @param indent_code A logical value indicating whether to indent the code in
#'   the tests.
#' @param add_testthat_boilerplate A logical value indicating whether to add a
#'   'test_that' boilerplate to the tests.
#' @param add_context_header A logical value indicating whether to add a
#'   'context' line to the header.
#'
#' @return A named list of content strings for the test files.
#' @keywords internal
process_testfiles <- function(testfiles,
                              indent_code = FALSE,
                              add_testthat_boilerplate = FALSE,
                              add_context_header = FALSE) {

  if (length(testfiles) == 0L) { return(list()) }

  paths <- names(testfiles)
  results <- lapply(seq_along(testfiles), function(i) {
    testfile <- testfiles[[i]]
    content <- lapply(testfile, function(x) {
      longtests <- lapply(x$longtests, function(l) l$raw)
      longtests <- prepare_tests(longtests)
      if (indent_code) { longtests <- indent_text(longtests) }
      tests_name <- x$functionname
      ifelse(
        add_testthat_boilerplate,
        paste0('test_that("', tests_name, '", {', "\n", longtests, "\n", "})\n"),
        paste0("## ", tests_name, "\n", longtests, "\n")
      )
    })

    content <- paste0(content, collapse = "\n\n")

    path_quoted <- ifelse(
      paths[i] == "<text>",
      paths[i],
      paste0('File R/', auto_quote(paths[i]), ': @longtests')
    )

    content_header <-
      paste0("# Generated by biocroxytest: do not edit by hand!\n\n")

    content_header <- ifelse(
      add_context_header,
      paste0(content_header, 'context("', path_quoted, '")', "\n\n"),
      paste0(content_header, "# ", path_quoted, "\n\n")
    )

    paste0(content_header, content)
  })

  names(results) <- paths

  results
}

#' Process blocks for 'longtests' roclet
#'
#' This is an internal function that processes a list of blocks for the
#' 'longtests' roclet. It first collects and annotates the 'rdname' of the
#' blocks. Then, it loops over each block and converts it to a testthat file
#' using the 'block_to_testthat' function. If the resulting testthat file has a
#' filename and contains 'longtests', it is added to the list of test files.
#' Finally, it processes the test files using the 'process_testfiles' function.
#'
#' @param blocks A list of roxygen2 blocks.
#' @param ... Additional arguments.
#'
#' @return A list with a single element 'longtests', which contains the
#'   processed test files.
#' @keywords internal
internal_longtests_roclet_process <- function(blocks, ...) {
  blocks <- collect_annotate_rdname(blocks)

  longtestfiles <- list()

  for (block in blocks) {
    testthat <- block_to_testthat(block)
    if (is.null(testthat$filename)) {
      next
    }
    if (length(testthat$longtests) > 0L) {
      longtestfiles[[testthat$filename]] <-
        c(longtestfiles[[testthat$filename]], list(testthat))
    }
  }

  list(longtests = process_testfiles(longtestfiles, indent_code = TRUE,
                                     add_testthat_boilerplate = TRUE,
                                     add_context_header = FALSE))
}

#' Write long test results to files
#'
#' This function takes three arguments: results, base_path, and prefix. The
#' function iterates over the elements of results and writes the content of each
#' element to a file in the directory specified by base_path and prefix. If the
#' file was not created by biocroxytest, the function informs the user that some
#' files were not removed.
#'
#' @param results A list of results.
#' @param base_path The base path where the files will be written.
#' @param prefix The prefix that will be used to name the files.
#'
#' @return A list of file paths.
#' @keywords internal
internal_longtests_roclet_output <- function(results,
                                             base_path,
                                             prefix = "test-biocroxytest") {

  paths <- names(results)
  for (i in seq_along(results)) {
    path <- file.path(base_path, paste0(prefix, paths[i]))

    if (!made_by_biocroxytest(path)) {
      cli::cli_inform(c(
        "!" = glue::glue(
          "Clean-up: Some files in {cli::col_blue('longtests/')} with the file ",
          "name pattern {cli::col_blue('test-biocroxytest-*.R')} was not created ",
          "by {cli::col_blue('biocroxytest')} (missing header), and hence was ",
          "not removed!"
        ),
        "*" = "Please be sure that this is intended."
      ))
      next
    }

    content <- results[[i]]

    writeLines(text = enc2utf8(content), con = path, useBytes = TRUE)
  }

  paths
}

#' Clean up longtest files generated by biocroxytest
#'
#' This is an internal function that cleans up test files generated by
#' biocroxytest. It checks each file in the provided list of test files. If a
#' file was generated by biocroxytest, it will be removed. If a file was not
#' generated by biocroxytest, a message will be printed to inform the user, and
#' the file will not be removed.
#'
#' @param testfiles A character vector representing the paths of the test files
#'   to check and clean.
#'
#' @return invisible
#' @keywords internal
internal_longtests_roclet_clean <- function(testfiles) {
  testfiles <- testfiles[!file.info(testfiles)$isdir]

  made_by_me <- vapply(testfiles, made_by_biocroxytest, logical(1))

  if (any(!made_by_me)) {
    cli::cli_inform(c(
      "!" = glue::glue(
        "Clean-up: Some files in {cli::col_blue('longtests/')} with the file ",
        "name pattern {cli::col_blue('test-biocroxytest-*.R')} was not created ",
        "by {cli::col_blue('biocroxytest')} (missing header), and hence was ",
        "not removed!"
      ),
      "*" = "Please be sure that this is intended."
    ))
  }

  unlink(testfiles[made_by_me])
}
