% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filterSpectriPy.R
\name{filterSpectriPy}
\alias{filterSpectriPy}
\alias{select_by_intensity}
\alias{select_by_mz}
\alias{remove_peaks_around_precursor_mz}
\alias{normalize_intensities}
\alias{filterSpectriPy,Spectra,filter_param-method}
\title{Filter Spectra using Python's matchms library}
\usage{
select_by_intensity(intensity_from = 10, intensity_to = 200)

select_by_mz(mz_from = 0, mz_to = 1000)

remove_peaks_around_precursor_mz(mz_tolerance = 17)

normalize_intensities()

\S4method{filterSpectriPy}{Spectra,filter_param}(object, param, mapping = spectraVariableMapping(), ...)
}
\arguments{
\item{intensity_from}{\code{numeric(1)}: Set lower threshold for peak intensity.
Default is 10.}

\item{intensity_to}{\code{numeric(1)}: Set upper threshold for peak intensity.
Default is 200.}

\item{mz_from}{\code{numeric(1)}: Set lower threshold for m/z peak positions.
Default is 0.}

\item{mz_to}{\code{numeric(1)}: Set upper threshold for m/z peak positions.
Default is 1000.}

\item{mz_tolerance}{\code{numeric(1)}: Tolerance of m/z values that are not
allowed to lie within the precursor mz. Default is 17 Da.}

\item{object}{A \code{\link[Spectra:Spectra]{Spectra::Spectra()}} object.}

\item{param}{one of parameter classes listed above (such as
\code{select_by_intensity()}) defining the filter/processing function in
Python and its parameters.}

\item{mapping}{named \code{character()} defining which spectra variables/metadata
should be converted between R and Python and how they should be renamed.
Defaults to \code{spectraVariableMapping()}. See \code{\link[=setSpectraVariableMapping]{setSpectraVariableMapping()}}
for more information.}

\item{...}{ignored.}
}
\value{
\code{filterSpectriPy()} returns a \code{Spectra} object on which the
filtering/processing function has been applied
}
\description{
The \code{filterSpectriPy()} function allows to filter/process a \code{Spectra} object
using the \code{select_by_intensity()}, \code{select_by_mz()},
\code{remove_peaks_around_precursor_mz()}, and \code{normalize_intensities()}
functions of the Python
\href{https://matchms.readthedocs.io/en/latest/api/matchms.filtering.html}{matchms.filtering}
module.

Selection and configuration of the algorithm can be performed with one of
the parameter objects (equivalent to \emph{matchms}' function names):
\itemize{
\item \code{select_by_intensity()}: Keeps only the peaks within defined intensity
range (keep if \code{intensity_from} >= intensity >= \code{intensity_to}). See also
the respective \href{https://matchms.readthedocs.io/en/latest/api/matchms.filtering.peak_processing.select_by_intensity.html}{documentation in \emph{matchms}}.
\item \code{select_by_mz()}: Keeps only the peaks between \code{mz_from} and \code{mz_to}
(keep if \code{mz_from} >= m/z >= \code{mz_to}). See also the respective
\href{https://matchms.readthedocs.io/en/latest/api/matchms.filtering.peak_processing.select_by_mz.html}{documentation in \emph{matchms}}.
\item \code{remove_peaks_around_precursor_mz()}: Removes the peaks that are within
\code{mz_tolerance} (in Da) of the precursor mz, excluding the precursor peak.
\item \code{normalize_intensities()}: Normalizes the intensities of peaks
(and losses) to unit height.
}
}
\note{
The first call to the \code{filterSpectriPy()} function can take longer because
the Python environment needs to be first set up.

\code{filterSpectriPy()} first translates the \code{Spectra} to Python, applies the
filter functions from the \emph{matchms} Python libraries and then translates
the filtered data back to a \code{Spectra} object. Thus, any spectra variables
other than those that are translated between R and Python will be lost
during the processing. Use \code{\link[=setSpectraVariableMapping]{setSpectraVariableMapping()}} to define which
spectra variables should be transferred/converted between R and Python.
See also examples below for more information.

The \code{\link[Spectra:Spectra]{Spectra::Spectra()}} object returned by \code{filterSpectriPy()} will
\strong{always} use an in-memory backend (i.e. the \code{\link[Spectra:MsBackend]{Spectra::MsBackendMemory()}}),
independently of the backend used by the backend used by the input
\code{Spectra}.
}
\examples{

library(Spectra)

## create some example Spectra
DF <- DataFrame(
    msLevel = c(2L, 2L, 2L),
    name = c("Caffeine", "Caffeine", "1-Methylhistidine"),
    precursorMz = c(195.0877, 195.0877, 170.0924)
)
DF$intensity <- list(
    c(340.0, 416, 2580, 412),
    c(388.0, 3270, 85, 54, 10111),
    c(3.407, 47.494, 3.094, 100.0, 13.240))
DF$mz <- list(
    c(135.0432, 138.0632, 163.0375, 195.0880),
    c(110.0710, 138.0655, 138.1057, 138.1742, 195.0864),
    c(109.2, 124.2, 124.5, 170.16, 170.52))
sps <- Spectra(DF)

## Filter: select_by_intensity
res <- filterSpectriPy(
    sps, select_by_intensity(intensity_from = 15, intensity_to = 300))
## Only mass peaks with intensities between the specified limits are
## retained
intensity(res)
## Compared to the original intensities
intensity(sps)

## Note that the spectra variable `"name"` was lost during conversion of
## the MS data between R and Python:
sps$name
any(spectraVariables(res) == "name")

## Only spectra variables defined by `spectraVariableMapping()` are
## converted and thus retained:
spectraVariableMapping()

## We can also pass a custom *spectra variable mapping* with the `mapping`
## parameter to the `filterSpectriPy()` function. Below we create such
## a mapping by adding the translation of a spectra variable `"name"` to
## a metadata name `"compound_name"` to the default spectra variable
## mapping `defaultSpectraVariableMapping()`.
map <- c(defaultSpectraVariableMapping(), name = "compound_name")
map

## Repeat the filtering operation passing this mapping information:
res <- filterSpectriPy(
    sps, select_by_intensity(intensity_from = 15, intensity_to = 300),
    mapping = map)
res$name

}
\seealso{
\itemize{
\item \code{\link[Spectra:filterMsLevel]{Spectra::filterIntensity()}}, \code{\link[Spectra:filterMsLevel]{Spectra::filterMzRange()}},
\code{\link[Spectra:addProcessing]{Spectra::scalePeaks()}} in the \code{Spectra} package for pure R
implementations of filtering/processing calculations.
\item \code{\link[=rspec_to_pyspec]{rspec_to_pyspec()}} or \code{\link[=pyspec_to_rspec]{pyspec_to_rspec()}} for the functions used to
translated the MS data between R and Python.
}
}
\author{
Thomas Naake
}
