##' plot_mappedReads_percent
##' 
##' A function to plot percent of mapped reads in total reads. Only RangedSummarizedExperiment 
##' object generated by recount package have total reads information to to this.
##' 
##'
##' 
##' @param expObj RangedSummarizedExperiment object generated by recount package.
##' @param groupVar variable name in colData(expObj) to be used to group the samples to make boxplot.
##' @return A barplot or boxplot.
##' @export
##' @examples 1
plot_mappedReads_percent = function(expObj, groupVar = NULL) {
	if (!("RangedSummarizedExperiment" %in% class(expObj)) || 
			!("reads_downloaded" %in% colnames(colData(expObj))) ||
			!("mapped_read_count" %in% colnames(colData(expObj)))
			) {
		last(paste0(
			"Only RangedSummarizedExperiment object generated by recount contains enough information for this function."
			))
	}
	if (is.null(groupVar)) {
		#barplot to show all samples
		dataForPlot = data.frame(
			sample = row.names(colData(expObj)),
			MappedReads = colData(expObj)$mapped_read_count,
			UnMappedReads = colData(expObj)$reads_downloaded - colData(expObj)$mapped_read_count
		)
		dataForPlot = dataForPlot %>% pivot_longer(
			cols = contains("Reads"),
			names_to = "Category",
			values_to = "Reads"
		)
		p = ggplot(dataForPlot, aes(x = sample, y = Reads, fill = Category)) +
			geom_bar(stat = "identity", position = "fill")
		p = p + theme(
			axis.text.x = element_text(
				angle = 90,
				vjust = 0.5,
				hjust = 1
			),
			text = element_text(size = 20)
		)
		
	} else {
		#boxplot to group samples by groupVar
		dataForPlot = data.frame(
			sample = row.names(colData(expObj)),
			Tissue = colData(expObj)[, groupVar],
			MappedReads = colData(expObj)$mapped_read_count,
			UnMappedReads = colData(expObj)$reads_downloaded - colData(expObj)$mapped_read_count
		)
		dataForPlot$MappedReadsPercent = dataForPlot$MappedReads / (dataForPlot$MappedReads +
																																	dataForPlot$UnMappedReads)
		p = ggplot(dataForPlot, aes(x = Tissue, y = MappedReadsPercent)) + geom_boxplot()
		p = p + theme(
			axis.text.x = element_text(
				angle = 90,
				vjust = 0.5,
				hjust = 1
			),
			text = element_text(size = 20)
		) + ylab("Percent of Mapped Reads")
		
	}
	return(p)
}

##' plot_gene_counts_range
##' 
##' A function to plot propotion of genes in different count range.
##' 
##'
##' 
##' @param expObj RangedSummarizedExperiment object or an expression matrix.
##' @param targetSize The target library size to scale to. Will not do scale if set as NULL.
##' @return A barplot.
##' @export
##' @examples 1
plot_gene_counts_range = function(expObj, targetSize = NULL) {
	if (!is.null(targetSize)) {
		#need scale data to targetSize
		if (("RangedSummarizedExperiment" %in% class(expObj))) {
			if ("mapped_read_count" %in% colnames(colData(expObj))) {
				#recount object
				expObj_scaled <- scale_counts(expObj, targetSize = targetSize)
				expObj_scaled_countTable = assay(expObj_scaled)
			} else {
				#ExpressionAtlas object
				expObj_countTable = assay(expObj)
				totalCounts = colSums(expObj_countTable)
				scaleFactor = targetSize / totalCounts
				expObj_scaled_countTable = round(t(t(expObj_countTable) * scaleFactor), 0)
			}
		} else { #expression matrix
			expObj_countTable=expObj
			totalCounts = colSums(expObj_countTable)
			scaleFactor = targetSize / totalCounts
			expObj_scaled_countTable = round(t(t(expObj_countTable) * scaleFactor), 0)
		}
	} else {
		if ((class(expObj) == "RangedSummarizedExperiment")) {
			expObj_scaled_countTable = assay(expObj)
		} else { #expression matrix
			expObj_scaled_countTable=expObj
		}
	}
	dataForPlot = data.frame(apply(expObj_scaled_countTable, 2, function(x)
		cut(
			x,
			c(0, 1, 5, 10, 20, Inf),
			include.lowest = TRUE,
			right = FALSE
		)))
	dataForPlot = dataForPlot %>% pivot_longer(cols = everything())
	dataForPlot$value = factor(dataForPlot$value,
														 levels = c("[0,1)", "[1,5)", "[5,10)", "[10,20)", "[20,Inf]"))
	p = ggplot(dataForPlot, aes(x = name, fill = value)) + geom_bar()
	p = p + theme(axis.text.x = element_text(
		angle = 90,
		vjust = 0.5,
		hjust = 1
	),
	text = element_text(size = 20))
	
	return(p)
	
}
