% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ProcessData.R
\name{RunCRE_HSAStringDB}
\alias{RunCRE_HSAStringDB}
\title{This function runs a causal relation engine by computing the Quaternary Dot
Product Scoring Statistic, Ternary Dot Product Scoring Statistic or the Enrichment test over the Homo
Sapien STRINGdb causal network (version 10 provided under the Creative Commons license: 
https://creativecommons.org/licenses/by/3.0/). Note that the user has the option of specifying other causal networks
with this function.}
\usage{
RunCRE_HSAStringDB(gene_expression_data, method = "Quaternary", 
                    fc.thresh = log2(1.3), pval.thresh = 0.05, 
                    only.significant.pvalues = FALSE, 
                    significance.level = 0.05,
                    epsilon = 1e-16, progressBar = TRUE, 
                    relations = NULL, entities = NULL)
}
\arguments{
\item{gene_expression_data}{A data frame for gene expression data. The \code{gene_expression_data} data frame must have three columns \code{entrez}, 
\code{fc} and \code{pvalue}. \code{entrez} denotes the entrez id of a given gene, \code{fc} denotes
the fold change of a gene, and \code{pvalue} denotes the p-value. The \code{entrez} column must be of type
integer or character, and the \code{fc} and \code{pvalue} columns must be numeric values.}

\item{method}{Choose one of \code{Quaternary}, \code{Ternary} or \code{Enrichment}. Default is \code{Quaternary}.}

\item{fc.thresh}{Threshold for fold change in \code{gene_expression_data} data frame. Any row in gene_expression_data with abosolute value of \code{fc}
smaller than \code{fc.thresh} will be ignored. Default value is \code{fc.thresh = log2(1.3)}.}

\item{pval.thresh}{Threshold for p-values in \code{gene_expression_data} data frame. All rows in \code{gene_expression_data} with p-values 
greater than \code{pval.thresh} will be ingnored. Default value is \code{pval.thresh = 0.05}.}

\item{only.significant.pvalues}{If \code{only.significant.pvalues = TRUE} then only p-values for statistically significant regulators
are computed otherwise uncomputed p-values are set to -1. The default value is \code{only.significant.pvalues = FALSE}.}

\item{significance.level}{When \code{only.significant.pvalues = TRUE}, only p-values which are less than or equal to 
\code{significance.level} are computed. The default value is \code{significance.level = 0.05}.}

\item{epsilon}{Threshold for probabilities of matrices. Default value is \code{threshold = 1e-16}.}

\item{progressBar}{Progress bar for the percentage of computed p-values for the regulators in the network. Default
value is \code{progressBar = TRUE}.}

\item{relations}{A data frame containing pairs of connected entities in a causal network,
and the type of causal relation between them. The data frame must have three columns with column names: \emph{srcuid}, 
\emph{trguid} and \emph{mode} respective of order. \emph{srcuid} stands for source entity, \emph{trguid} stands for 
target entity and \emph{mode} stands for the type of relation between \emph{srcuid} and \emph{trguid}. The relation 
has to be one of \emph{+1} for \emph{upregulation}, \emph{-1} for \emph{downregulation} or \emph{0} for regulation without
specified direction of regulation. All three columns must be of type integer. Default value is \code{relations = NULL}.}

\item{entities}{A data frame of mappings for all entities present in data frame \emph{relations}. \emph{entities} must contain
four columns: \emph{uid}, \emph{id}, \emph{symbol} and \emph{type} respective of order. \emph{uid} must be 
of type integer and \emph{id}, \emph{symbol} and \emph{type} must be of type character. \emph{uid} includes every source and target 
node in the network (i.e \emph{relations}),
\emph{id} is the id of \emph{uid} (e.g entrez id of an mRNA), \emph{symbol} is the symbol of \emph{id} and \emph{type} 
is the type of entity of \emph{id} (e.g mRNA, protein, drug or compound). Default value is \code{entities = NULL}.}
}
\value{
This function returns a data frame containing parameters concerning the method used. The p-values of each
        of the regulators is also computed, and the data frame
        is in increasing order of p-values of the goodness of fit score for the given regulators. The column
        names of the data frame are:
        
\itemize{        
\item  \code{uid} The regulator in the causal network.
\item \code{symbol} Symbol of the regulator. 
\item \code{regulation} Direction of regulation of the regulator.
\item \code{correct.pred} Number of correct predictions in \code{gene_expression_data} when compared to predictions made
                    by the network.
\item \code{incorrect.pred} Number of incorrect predictions in \code{gene_expression_data} when compared to predictions made
                    by the network.
\item \code{score} The number of correct predictions minus the number of incorrect predictions. 
\item \code{total.reachable} Total Number of children of the given regulator.
\item \code{significant.reachable} Number of children of the given regulator that are also present 
                                   in \code{gene_expression_data}.
\item \code{total.ambiguous} Total number of children of the given regulator which are regulated by the given regulator without
                             knowing the direction of regulation.
\item \code{significant.ambiguous} Total number of children of the given regulator which are regulated by the given regulator without
                             knowing the direction of regulation and are also present in \code{gene_expression_data}.  
\item \code{unknown} Number of target nodes in the causal network which do not interact with the given regulator.
\item \code{pvalue} P-value of the score computed according to the selected method. If \code{only.significant.pvalues = TRUE}
                    and the \code{pvalue} of the regulator is greater than \code{significance.level}, then
                    the p-value is not computed and is set to a value of -1.
}
}
\description{
This function runs a causal relation engine by computing the Quaternary Dot
             Product Scoring Statistic, Ternary Dot Product Scoring Statistic or the Enrichment test over the Homo
             Sapien STRINGdb causal network (version 10 provided under the Creative Commons license: 
             https://creativecommons.org/licenses/by/3.0/). Note that the user has the option of specifying other causal networks
             with this function.
}
\examples{

# Get gene expression data
e2f3 <- system.file("extdata", "e2f3_sig.txt", package = "QuaternaryProd")
e2f3 <- read.table(e2f3, sep = "\\t", header = TRUE, stringsAsFactors = FALSE)

# Rename column names appropriately and remove duplicated entrez ids
names(e2f3) <- c("entrez", "pvalue", "fc")
e2f3 <- e2f3[!duplicated(e2f3$entrez),]

# Compute the Quaternary Dot Product Scoring statistic for statistically significant
# regulators in the STRINGdb network
enrichment_results <- RunCRE_HSAStringDB(e2f3, method = "Enrichment",
                             fc.thresh = log2(1.3), pval.thresh = 0.05,
                             only.significant.pvalues = TRUE)
enrichment_results[1:4, c("uid","symbol","regulation","pvalue")]

}
\references{
Carl Tony Fakhry, Parul Choudhary, Alex Gutteridge, Ben Sidders, Ping Chen, Daniel Ziemek, and
            Kourosh Zarringhalam. Interpreting transcriptional changes using causal graphs: new methods and
            their practical utility on public networks. BMC Bioinformatics, 17:318, 2016. ISSN 1471-2105.
            doi: 10.1186/s12859-016-1181-8.
           
            Franceschini, A (2013). STRING v9.1: protein-protein interaction networks, with increased coverage 
            and integration. In:'Nucleic Acids Res. 2013 Jan;41(Database issue):D808-15. doi: 10.1093/nar/gks1094. 
            Epub 2012 Nov 29'.
}
\author{
Carl Tony Fakhry, Ping Chen and Kourosh Zarringhalam
}
