% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AggLimmaWorkflow.R
\name{AggLimmaWorkflow}
\alias{AggLimmaWorkflow}
\title{Aggregation-based LIMMA workflow}
\usage{
AggLimmaWorkflow(
  dat,
  contrasts.par,
  group,
  pep_mapping_tbl,
  covar = NULL,
  method = c("sum", "robreg"),
  logged = c(TRUE, FALSE),
  npep.trend = FALSE,
  eb = TRUE
)
}
\arguments{
\item{dat}{a dataframe or matrix of peptide abundance, or a SummarizedExperiment object where
grouping and peptide-protein mapping are provided in colData and rowData, respectively.}

\item{contrasts.par}{group levels to be compared separated by dash (e.g., "B-A"
if group B is to be compared against group A)}

\item{group}{a vector of group levels corresponding to each sample. Alternatively, it can be the
column name of the group in colData if dat is a SummarizedExperiment object.}

\item{pep_mapping_tbl}{a table mapping peptides to proteins. Alternatively, it can be the
column name of the protein in rowData if dat is a SummarizedExperiment object.}

\item{covar}{covariate matrix. Alternatively, it can be the column names of the covariates
in colData if dat is a SummarizedExperiment object.}

\item{method}{method of aggregation. Options including "sum" (summed peptide intensity)
and "robreg" (robust regression with M-Estimation).}

\item{logged}{Boolean variable indicating whether abundance data have been
log-transformed}

\item{npep.trend}{logical, should a number-of-peptide-trend be allowed for the prior
variance? Default is constant prior variance.}

\item{eb}{logical, whether to output the result from the empirical Bayes or ordinary approach.}
}
\value{
\code{AggLimmaWorkflow} returns a dataframe containing the following columns
\item{feature}{unique protein identifier}
\item{logFC}{log2 fold change}
\item{t}{t-statistic}
\item{P.Value}{raw p-value}
\item{adj.P.Val}{p-value adjusted via the Benjamini-Hochberg method}
\item{B}{B-statistic (empirical Bayes only)}
}
\description{
Given peptide abundance and assignment of peptide sequences to proteins,
execute the aggregation-based LIMMA workflow to compute the log2 fold change,
p-value, and adjusted p-value of all proteins identified.
}
\examples{
# Generate random peptide data
dat <- 2^matrix(rnorm(3000), ncol = 6)
colnames(dat) <- paste0("Sample", 1:6)
rownames(dat) <- paste0("Peptide", 1:500)

# Generate peptide mapping table
pep_mapping_tbl <- data.frame(peptide = paste0("Peptide", 1:500),
protein = paste0("Protein", rep(1:100, each = 5)))

# Generate groups and contrasts
group <- c(rep("A", 3), rep("B", 3))
contrasts.par <- "B-A"

AggLimmaWorkflow(dat, contrasts.par = contrasts.par,
group = group,
pep_mapping_tbl = pep_mapping_tbl,
method = "sum",
logged = FALSE)

# Store data as a SummarizedExperiment object; add covariates
library(tibble)
library(SummarizedExperiment)
colData <- data.frame(sample = LETTERS[seq_along(group)], group = group, 
sex = c("M", "F", "M", "F", "F", "M"), age = 1:6) |> 
column_to_rownames(var = "sample")
rowData <- pep_mapping_tbl |> column_to_rownames(var = "peptide")
dat.nn <- dat
rownames(dat.nn) <- NULL
colnames(dat.nn) <- NULL
dat.se <- SummarizedExperiment(assays = list(int = dat.nn), colData = colData, rowData = rowData)

AggLimmaWorkflow(dat.se, contrasts.par = contrasts.par,
group = "group",
covar = c("sex", "age"),
pep_mapping_tbl = "protein",
method = "sum",
logged = FALSE)

}
\references{
Ritchie, ME, Phipson, B, Wu, D, Hu, Y, Law, CW, Shi, W, and Smyth, GK (2015). limma powers differential expression analyses for RNA-sequencing and microarray studies. \emph{Nucleic Acids Research} 43, e47.
}
\author{
Junmin Wang
}
