#!/usr/bin/env Rscript

#
#  This file is part of the `OmnipathR` R package
#
#  Copyright
#  2018-2025
#  Saez Lab, Uniklinik RWTH Aachen, Heidelberg University
#
#  File author(s): Alberto Valdeolivas
#                  Dénes Türei (turei.denes@gmail.com)
#                  Attila Gábor
#
#  Distributed under the MIT (Expat) License.
#  See accompanying file `LICENSE` or find a copy at
#      https://directory.fsf.org/wiki/License:Expat
#
#  Website: https://r.omnipathdb.org/
#  Git repo: https://github.com/saezlab/OmnipathR
#


#' Enzyme-substrate (PTM) relationships from OmniPath
#'
#' Imports the enzyme-substrate (more exactly, enzyme-PTM) relationship
#' database from \url{https://omnipathdb.org/enzsub}. These are mostly
#' kinase-substrate relationships, with some acetylation and other types of
#' PTMs.
#'
#' @return A data frame of enzymes and their PTM substrates.
#'
#' @param ... Arguments passed to \code{\link{omnipath_query}}.
#' @inheritDotParams omnipath_query -query_type -datasets -types -json_param
#'     -add_counts -references_by_resource
#'
#' @examples
#' enzsub <- enzyme_substrate(
#'     resources = c("PhosphoSite", "SIGNOR"),
#'     organism = 9606
#' )
#'
#' @seealso \itemize{
#'     \item{\code{\link{enzsub_resources}}}
#'     \item{\code{\link{omnipath_interactions}}}
#'     \item{\code{\link{enzsub_graph}}}
#'     \item{\code{\link{print_interactions}}}
#'     \item{\code{\link{query_info}}}
#'     \item{\code{\link{omnipath_query}}}
#' }
#'
#' @importFrom rlang exec !!!
#' @export
#' @aliases import_omnipath_enzsub
enzyme_substrate <- function(...){

    args <- omnipath_args(list(...), query_type = 'enzsub')

    exec(omnipath_query, !!!args)

}


# Aliases (old names) to be Deprecated
#' @rdname complexes
#' @param ... Passed to \code{enzyme_substrate}.
#' @export
#'
#' @noRd
import_omnipath_enzsub <- function(...){
    .Deprecated('enzyme_substrate')
    enzyme_substrate(...)
}


#' Retrieves a list of enzyme-substrate resources available in OmniPath
#'
#' Get the names of the enzyme-substrate relationship resources available
#' in \url{https://omnipathdb.org/enzsub}
#'
#' @param dataset ignored for this query type
#' @return character vector with the names of the enzyme-substrate resources
#' @export
#'
#' @examples
#' enzsub_resources()
#'
#' @seealso \itemize{
#'     \item{\code{\link{resources}}}
#'     \item{\code{\link{enzyme_substrate}}}
#' }
#'
#' @aliases get_enzsub_resources
enzsub_resources <- function(dataset = NULL){

    return(resources(query_type = 'enzsub', datasets = dataset))

}


# Aliases (old names) to be Deprecated
#' @rdname complexes
#' @param ... Passed to \code{enzsub_resources}.
#' @export
#'
#' @noRd
get_enzsub_resources <- function(...){
    .Deprecated('enzsub_resources')
    enzsub_resources(...)
}


#' Causal effect enzyme-PTM interactions
#'
#' Enzyme-substrate data does not contain sign (activation/inhibition), we
#' generate this information based on the interaction network.
#'
#' @param enzsub Enzyme-substrate data frame generated by
#'     \code{\link{enzyme_substrate}}
#' @param interactions interaction data frame generated by an OmniPath
#'     interactions query: \code{\link{omnipath-interactions}}
#'
#' @return Data frame of enzyme-substrate relationships with is_inhibition
#'     and is_stimulation columns.
#'
#' @export
#'
#' @examples
#' enzsub <- enzyme_substrate(resources = c("PhosphoSite", "SIGNOR"))
#' interactions <- omnipath_interactions()
#' enzsub <- signed_ptms(enzsub, interactions)
#'
#' @seealso \itemize{
#'     \item{\code{\link{enzyme_substrate}}}
#'     \item{\code{\link{omnipath-interactions}}}
#' }
#' @aliases get_signed_ptms
signed_ptms <- function(
    enzsub = enzyme_substrate(),
    interactions = omnipath_interactions()
){

    signed <- merge(
        enzsub,
        interactions[,
            c('source', 'target', 'is_stimulation', 'is_inhibition')
        ],
        by.x = c('enzyme', 'substrate'),
        by.y = c('source', 'target'),
        all.x = TRUE
    )

    return(signed)
}
