% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aggregate.R, R/aggregate_by_matrix.R,
%   R/aggregate_by_vector.R
\name{aggregate}
\alias{aggregate}
\alias{aggregate_by_vector}
\alias{colMeansMat}
\alias{colSumsMat}
\alias{aggregate_by_matrix}
\title{Aggreagate quantitative features}
\usage{
colMeansMat(x, MAT, na.rm = FALSE)

colSumsMat(x, MAT, na.rm = FALSE)

aggregate_by_matrix(x, MAT, FUN, ...)

aggregate_by_vector(x, INDEX, FUN, ...)
}
\arguments{
\item{x}{A \code{matrix} of mode \code{numeric} or an \code{HDF5Matrix} object of
type \code{numeric}.}

\item{MAT}{An adjacency matrix that defines peptide-protein
relations with \code{nrow(MAT) == nrow(x)}: a non-missing/non-null
value at position (i,j) indicates that peptide i belong to
protein j. This matrix is tyically binary but can also contain
weighted relations.}

\item{na.rm}{A \code{logical(1)} indicating whether the missing values
(including NaN) should be omitted from the calculations or not.
Defaults to \code{FALSE}.}

\item{FUN}{A \code{function} to be applied to the subsets of \code{x}.}

\item{...}{Additional arguments passed to \code{FUN}.}

\item{INDEX}{A \code{vector} or \code{factor} of length \code{nrow(x)}.}
}
\value{
\code{\link[=aggregate_by_matrix]{aggregate_by_matrix()}} returns a \code{matrix} (or \code{Matrix})
of dimensions \code{ncol(MAT)} and \verb{ncol(x), with }dimnames\verb{equal to}colnames(x)\code{and}rownames(MAT)`.

\code{\link[=aggregate_by_vector]{aggregate_by_vector()}} returns a new \code{matrix} (if \code{x} is
a \code{matrix}) or \code{HDF5Matrix} (if \code{x} is an \code{HDF5Matrix})
of dimensions \code{length(INDEX)} and \verb{ncol(x), with }dimnames\verb{ equal to}colnames(x)\code{and}INDEX`.
}
\description{
These functions take a matrix of quantitative features \code{x} and
aggregate the features (rows) according to either a vector (or
factor) \code{INDEX} or an adjacency matrix \code{MAT}. The aggregation
method is defined by function \code{FUN}.

Adjacency matrices are an elegant way to explicitly encode for
shared peptides (see example below) during aggregation.
}
\section{Vector-based aggregation functions}{


When aggregating with a vector/factor, user-defined functions
must return a vector of length equal to \code{ncol(x)} for each level
in \code{INDEX}. Examples thereof are:
\itemize{
\item \code{\link[=medianPolish]{medianPolish()}} to fits an additive model (two way
decomposition) using Tukey's median polish procedure using
\code{\link[stats:medpolish]{stats::medpolish()}};
\item \code{\link[=robustSummary]{robustSummary()}} to calculate a robust aggregation using
\code{\link[MASS:rlm]{MASS::rlm()}};
\item \code{\link[base:colSums]{base::colMeans()}} to use the mean of each column;
\item \code{\link[base:colSums]{base::colSums()}} to use the sum of each column;
\item \code{\link[matrixStats:rowMedians]{matrixStats::colMedians()}} to use the median of each column.
}
}

\section{Matrix-based aggregation functions}{


When aggregating with an adjacency matrix, user-defined
functions must return a new matrix. Examples thereof are:
\itemize{
\item \code{colSumsMat(x, MAT)} aggregates by the summing the peptide intensities
for each protein. Shared peptides are re-used multiple times.
\item \code{colMeansMat(x, MAT)} aggregation by the calculating the mean of
peptide intensities. Shared peptides are re-used multiple
times.
}
}

\section{Handling missing values}{


By default, missing values in the quantitative data will propagate
to the aggregated data. You can provide \code{na.rm = TRUE} to most
functions listed above to ignore missing values, except for
\code{robustSummary()} where you should supply \code{na.action = na.omit}
(see \code{?MASS::rlm}).
}

\examples{

x <- matrix(c(10.39, 17.16, 14.10, 12.85, 10.63, 7.52, 3.91,
              11.13, 16.53, 14.17, 11.94, 11.51, 7.69, 3.97,
              11.93, 15.37, 14.24, 11.21, 12.29, 9.00, 3.83,
              12.90, 14.37, 14.16, 10.12, 13.33, 9.75, 3.81),
            nrow = 7,
            dimnames = list(paste0("Pep", 1:7), paste0("Sample", 1:4)))
x

## -------------------------
## Aggregation by vector
## -------------------------

(k <- paste0("Prot", c("B", "E", "X", "E", "B", "B", "E")))

aggregate_by_vector(x, k, colMeans)
aggregate_by_vector(x, k, robustSummary)
aggregate_by_vector(x, k, medianPolish)

## -------------------------
## Aggregation by matrix
## -------------------------

adj <- matrix(c(1, 0, 0, 1, 1, 1, 0, 0,
                1, 0, 1, 0, 0, 1, 0, 0,
                1, 0, 0, 0, 1),
              nrow = 7,
              dimnames = list(paste0("Pep", 1:7),
                              paste0("Prot", c("B", "E", "X"))))
adj

## Peptide 4 is shared by 2 proteins (has a rowSums of 2),
## namely proteins B and E
rowSums(adj)

aggregate_by_matrix(x, adj, colSumsMat)
aggregate_by_matrix(x, adj, colMeansMat)

## ---------------
## Missing values
## ---------------

x <- matrix(c(NA, 2:6), ncol = 2,
            dimnames = list(paste0("Pep", 1:3),
                            c("S1", "S2")))
x

## simply use na.rm = TRUE to ignore missing values
## during the aggregation

(k <- LETTERS[c(1, 1, 2)])
aggregate_by_vector(x, k, colSums)
aggregate_by_vector(x, k, colSums, na.rm = TRUE)

(adj <- matrix(c(1, 1, 0, 0, 0, 1), ncol = 2,
               dimnames = list(paste0("Pep", 1:3),
                           c("A", "B"))))
aggregate_by_matrix(x, adj, colSumsMat, na.rm = FALSE)
aggregate_by_matrix(x, adj, colSumsMat, na.rm = TRUE)

}
\seealso{
Other Quantitative feature aggregation: 
\code{\link{colCounts}()},
\code{\link{medianPolish}()},
\code{\link{robustSummary}()}
}
\author{
Laurent Gatto and Samuel Wieczorek (aggregation from an
adjacency matrix).
}
\concept{Quantitative feature aggregation}
